#!/bin/bash

# BrightTally WordPress Plugin WP CLI Helper Script
# Usage: ./wp-cli.sh [WP-CLI-COMMAND]

set -e

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

print_status() {
    echo -e "${GREEN}[INFO]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

print_header() {
    echo -e "${BLUE}$1${NC}"
}

# Check if WordPress container is running
if ! docker ps --format "table {{.Names}}" | grep -q "brighttally-wp-test"; then
    print_error "WordPress container is not running. Start it first with: ./start-wordpress.sh start"
    exit 1
fi

# If no arguments provided, show help
if [ $# -eq 0 ]; then
    print_header "🚀 BrightTally WordPress WP CLI Helper"
    echo ""
    echo "Usage: $0 [WP-CLI-COMMAND]"
    echo ""
    echo "Examples:"
    echo "  $0 --version                    # Check WP CLI version"
    echo "  $0 plugin list                  # List all plugins"
    echo "  $0 plugin activate bright-tally # Activate BrightTally plugin"
    echo "  $0 plugin status bright-tally   # Check plugin status"
    echo "  $0 plugin deactivate bright-tally # Deactivate plugin"
    echo "  $0 core version                 # Check WordPress version"
    echo ""
    echo "Config Commands (Edit wp-config.php easily):"
    echo "  $0 config set BRIGHT_TALLY_API_URL 'http://host.docker.internal:8000' --type=constant"
    echo "  $0 config get BRIGHT_TALLY_API_URL  # Check current value"
    echo "  $0 config delete BRIGHT_TALLY_API_URL  # Remove constant"
    echo "  $0 config get WP_DEBUG          # Check debug settings"
    echo ""
    echo "Other Commands:"
    echo "  $0 db query \"SELECT * FROM wp_posts LIMIT 5\" # Run database query"
    echo "  $0 eval \"phpinfo();\"           # Run PHP code"
    echo ""
    echo "Debug Commands:"
    echo "  $0 eval \"var_dump(error_get_last());\" # Check last error"
    echo "  $0 eval \"error_reporting(E_ALL); ini_set('display_errors', 1);\" # Enable error display"
    echo ""
    echo "To see all available commands:"
    echo "  $0 help"
    echo ""
    exit 0
fi

# Run WP CLI command
print_status "Running: wp $*"
docker exec brighttally-wp-test wp "$@" --allow-root
