#!/bin/bash

# BrightTally WordPress Plugin Testing Setup Script
# This script sets up a local WordPress environment for testing the plugin

set -e

echo "🚀 Setting up BrightTally WordPress Plugin Testing Environment"
echo "=============================================================="

# Check if Docker is installed
if ! command -v docker &> /dev/null; then
    echo "❌ Docker is not installed. Please install Docker first."
    echo "   Visit: https://docs.docker.com/get-docker/"
    exit 1
fi

# Check if Docker Compose is installed
if ! command -v docker compose &> /dev/null; then
    echo "❌ Docker Compose is not installed. Please install Docker Compose first."
    echo "   Visit: https://docs.docker.com/compose/install/"
    exit 1
fi

echo "✅ Docker and Docker Compose are installed"

# Create .env file if it doesn't exist
if [ ! -f .env ]; then
    echo "📝 Creating .env file..."
    cat > .env << EOF
# BrightTally WordPress Testing Environment
WORDPRESS_DB_HOST=db:3306
WORDPRESS_DB_USER=wordpress
WORDPRESS_DB_PASSWORD=wordpress
WORDPRESS_DB_NAME=wordpress
WORDPRESS_DEBUG=1

MYSQL_DATABASE=wordpress
MYSQL_USER=wordpress
MYSQL_PASSWORD=wordpress
MYSQL_ROOT_PASSWORD=rootpassword
EOF
    echo "✅ .env file created"
fi

# Start the environment
echo "🐳 Starting Docker containers..."
docker compose up -d

# Wait for WordPress to be ready
echo "⏳ Waiting for WordPress to be ready..."
sleep 10

# Check if WordPress is accessible
echo "🔍 Checking WordPress accessibility..."
if curl -s http://localhost:8080 > /dev/null; then
    echo "✅ WordPress is accessible at http://localhost:8080"
else
    echo "⚠️  WordPress might still be starting up. Please wait a moment and try again."
fi

echo ""
echo "🎉 Setup Complete!"
echo "=================="
echo ""
echo "📋 Next Steps:"
echo "1. Open http://localhost:8080 in your browser"
echo "2. Complete the WordPress installation wizard"
echo "3. Go to Plugins and activate 'BrightTally'"
echo "4. Go to BrightTally > Settings to configure the plugin"
echo ""
echo "🔗 Useful Links:"
echo "   WordPress: http://localhost:8080"
echo "   phpMyAdmin: http://localhost:8081"
echo ""
echo "🛠  Development Commands:"
echo "   Stop environment: docker-compose down"
echo "   View logs: docker-compose logs -f"
echo "   Restart: docker-compose restart"
echo "   Fresh start: docker-compose down -v && docker-compose up -d"
echo ""
echo "📚 For more information, see README-TESTING.md"
