<?php
/**
 * The main plugin class
 *
 * @package Bright_Tally
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Bright_Tally {
    
    /**
     * The loader that's responsible for maintaining and registering all hooks
     *
     * @since 1.0.0
     * @var Bright_Tally_Loader
     */
    protected $loader;
    
    /**
     * The unique identifier of this plugin
     *
     * @since 1.0.0
     * @var string
     */
    protected $plugin_name;
    
    /**
     * The current version of the plugin
     *
     * @since 1.0.0
     * @var string
     */
    protected $version;
    
    /**
     * Define the core functionality of the plugin
     *
     * @since 1.0.0
     */
    public function __construct() {
        if (defined('BRIGHT_TALLY_VERSION')) {
            $this->version = BRIGHT_TALLY_VERSION;
        } else {
            $this->version = '1.0.0';
        }
        
        $this->plugin_name = 'bright-tally';
        
        $this->load_dependencies();
        $this->check_updates();
        $this->set_locale();
        $this->define_admin_hooks();
        $this->define_public_hooks();
        $this->define_update_notices();
    }
    
    /**
     * Load the required dependencies for this plugin
     *
     * @since 1.0.0
     */
    private function load_dependencies() {
        require_once BRIGHT_TALLY_PLUGIN_DIR . 'includes/class-bright-tally-loader.php';
        require_once BRIGHT_TALLY_PLUGIN_DIR . 'includes/class-bright-tally-i18n.php';
        require_once BRIGHT_TALLY_PLUGIN_DIR . 'includes/class-bright-tally-url-helper.php';
        require_once BRIGHT_TALLY_PLUGIN_DIR . 'includes/class-bright-tally-token-manager.php';
        require_once BRIGHT_TALLY_PLUGIN_DIR . 'includes/class-bright-tally-oauth.php';
        require_once BRIGHT_TALLY_PLUGIN_DIR . 'includes/class-bright-tally-updater.php';
        require_once BRIGHT_TALLY_PLUGIN_DIR . 'admin/class-bright-tally-admin.php';
        require_once BRIGHT_TALLY_PLUGIN_DIR . 'public/class-bright-tally-public.php';
        require_once BRIGHT_TALLY_PLUGIN_DIR . 'includes/class-bright-tally-api.php';
        require_once BRIGHT_TALLY_PLUGIN_DIR . 'includes/class-bright-tally-shortcode.php';
        
        $this->loader = new Bright_Tally_Loader();
    }
    
    /**
     * Check for plugin updates and run migrations if needed
     *
     * @since 1.0.0
     */
    private function check_updates() {
        Bright_Tally_Updater::check_updates();
    }
    
    /**
     * Register hooks for update notices
     *
     * @since 1.0.0
     */
    private function define_update_notices() {
        $this->loader->add_action('admin_notices', 'Bright_Tally_Updater', 'display_update_notices');
    }
    
    /**
     * Define the locale for this plugin for internationalization
     *
     * @since 1.0.0
     */
    private function set_locale() {
        $plugin_i18n = new Bright_Tally_i18n();
        $this->loader->add_action('plugins_loaded', $plugin_i18n, 'load_plugin_textdomain');
    }
    
    /**
     * Register all of the hooks related to the admin area functionality
     *
     * @since 1.0.0
     */
    private function define_admin_hooks() {
        $plugin_admin = new Bright_Tally_Admin($this->get_plugin_name(), $this->get_version());
        
        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_styles');
        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts');
        $this->loader->add_action('admin_menu', $plugin_admin, 'add_plugin_admin_menu');
        $this->loader->add_action('admin_init', $plugin_admin, 'register_settings');
    }
    
    /**
     * Register all of the hooks related to the public-facing functionality
     *
     * @since 1.0.0
     */
    private function define_public_hooks() {
        $plugin_public = new Bright_Tally_Public($this->get_plugin_name(), $this->get_version());
        
        $this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_styles');
        $this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_scripts');
        
        // Register shortcode
        $shortcode = new Bright_Tally_Shortcode();
        $this->loader->add_action('init', $shortcode, 'register_shortcode');
        
        // Register Gutenberg block
        $this->loader->add_action('init', $this, 'register_gutenberg_block');
        
        // Register REST API endpoint for polls
        $this->loader->add_action('rest_api_init', $this, 'register_rest_routes');
    }
    
    /**
     * Register Gutenberg block
     *
     * @since 1.0.0
     */
    public function register_gutenberg_block() {
        // Check if Gutenberg is available
        if (!function_exists('register_block_type')) {
            return;
        }
        
        // Register block using block.json
        $block_path = BRIGHT_TALLY_PLUGIN_DIR . 'blocks/bright-tally-block';
        
        // Check if block.json exists
        if (!file_exists($block_path . '/block.json')) {
            return;
        }
        
        // Register block
        register_block_type(
            $block_path,
            array(
                'render_callback' => array($this, 'render_gutenberg_block'),
            )
        );
    }
    
    /**
     * Render Gutenberg block on frontend
     *
     * @since 1.0.0
     * @param array $attributes Block attributes
     * @return string Rendered shortcode
     */
    public function render_gutenberg_block($attributes) {
        $shortcode = new Bright_Tally_Shortcode();
        return $shortcode->render_shortcode($attributes);
    }
    
    /**
     * Register REST API routes
     *
     * @since 1.0.0
     */
    public function register_rest_routes() {
        register_rest_route('bright-tally/v1', '/polls', array(
            'methods' => 'GET',
            'callback' => array($this, 'rest_get_polls'),
            'permission_callback' => array($this, 'rest_check_permission'),
        ));
    }
    
    /**
     * REST API callback to get polls
     *
     * @since 1.0.0
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function rest_get_polls($request) {
        $api = new Bright_Tally_API();
        
        if (!$api->is_configured()) {
            return new WP_Error('not_configured', __('API not configured', 'bright-tally'), array('status' => 400));
        }
        
        $polls = $api->get_polls();
        
        if (is_wp_error($polls)) {
            return new WP_Error('api_error', $polls->get_error_message(), array('status' => 500));
        }
        
        return rest_ensure_response(array(
            'success' => true,
            'data' => $polls,
        ));
    }
    
    /**
     * Check REST API permission
     *
     * @since 1.0.0
     * @return bool
     */
    public function rest_check_permission() {
        return current_user_can('manage_options');
    }
    
    /**
     * Run the loader to execute all of the hooks with WordPress
     *
     * @since 1.0.0
     */
    public function run() {
        $this->loader->run();
    }
    
    /**
     * The name of the plugin used to uniquely identify it within the context of
     * WordPress and to define internationalization functionality
     *
     * @since 1.0.0
     * @return string The name of the plugin
     */
    public function get_plugin_name() {
        return $this->plugin_name;
    }
    
    /**
     * The reference to the class that orchestrates the hooks with the plugin
     *
     * @since 1.0.0
     * @return Bright_Tally_Loader Orchestrates the hooks of the plugin
     */
    public function get_loader() {
        return $this->loader;
    }
    
    /**
     * Retrieve the version number of the plugin
     *
     * @since 1.0.0
     * @return string The version number of the plugin
     */
    public function get_version() {
        return $this->version;
    }
}
