<?php
/**
 * URL Helper for environment-based API URL configuration
 *
 * @package Bright_Tally
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Bright_Tally_URL_Helper {
    
    /**
     * Get the API URL based on environment
     * 
     * Priority:
     * 1. WordPress option (user setting)
     * 2. BRIGHT_TALLY_API_URL constant (wp-config.php)
     * 3. Environment detection (localhost, Docker, etc.)
     * 4. Production default
     *
     * @since 1.0.0
     * @return string API base URL
     */
    public static function get_api_url() {
        // 0. If we're in Docker, prioritize container URL (even over constants)
        // This ensures server-side API calls use container-to-container communication
        if (self::is_docker_container()) {
            // Check for container URL constant first
            if (defined('BRIGHT_TALLY_CONTAINER_URL') && !empty(BRIGHT_TALLY_CONTAINER_URL)) {
                return BRIGHT_TALLY_CONTAINER_URL;
            }
            // Use service name for container-to-container communication
            // Service name is 'laravel.test' in docker-compose.yml
            $container_url = self::detect_development_url();
            if ($container_url && (
                strpos($container_url, 'laravel.test') !== false ||
                strpos($container_url, 'host.docker.internal') !== false
            )) {
                return $container_url;
            }
        }
        
        // 1. Check if user has set a custom URL in settings
        $saved_url = get_option('bright_tally_api_url', '');
        if (!empty($saved_url)) {
            return $saved_url;
        }
        
        // 2. Check for WordPress constant (for browser URLs, but not for container-to-container)
        // Only use constant if we're NOT in Docker, or if explicitly set for container use
        if (defined('BRIGHT_TALLY_API_URL') && !empty(BRIGHT_TALLY_API_URL)) {
            // If we're in Docker, the constant is likely meant for browser URLs
            // Check if there's a container-specific URL we should use instead
            if (self::is_docker_container()) {
                // Use container URL instead of browser URL
                $container_url = self::detect_development_url();
                if ($container_url) {
                    return $container_url;
                }
            }
            return BRIGHT_TALLY_API_URL;
        }
        
        // 3. Auto-detect development environment
        $dev_url = self::detect_development_url();
        if ($dev_url) {
            return $dev_url;
        }
        
        // 4. Default to production
        return 'https://brighttally.com'; // Default to production
    }
    
    /**
     * Detect if we're in a development environment and return appropriate URL
     *
     * @since 1.0.0
     * @return string|false Development URL or false if not in dev
     */
    private static function detect_development_url() {
        // Check for common development indicators
        $is_local = (
            // WordPress debug mode
            (defined('WP_DEBUG') && WP_DEBUG) ||
            // Localhost
            (isset($_SERVER['HTTP_HOST']) && (
                strpos($_SERVER['HTTP_HOST'], 'localhost') !== false ||
                strpos($_SERVER['HTTP_HOST'], '127.0.0.1') !== false ||
                strpos($_SERVER['HTTP_HOST'], '.local') !== false ||
                strpos($_SERVER['HTTP_HOST'], '.test') !== false ||
                strpos($_SERVER['HTTP_HOST'], '.docker') !== false
            )) ||
            // Docker Desktop (common patterns)
            (isset($_SERVER['HTTP_HOST']) && strpos($_SERVER['HTTP_HOST'], 'localhost:') !== false) ||
            // Environment variable
            (defined('WP_ENVIRONMENT_TYPE') && WP_ENVIRONMENT_TYPE === 'local')
        );
        
        if ($is_local) {
            // Check for custom dev URL constant first
            if (defined('BRIGHT_TALLY_DEV_URL') && !empty(BRIGHT_TALLY_DEV_URL)) {
                return BRIGHT_TALLY_DEV_URL;
            }
            
            // Check if we're running in Docker (WordPress container)
            // Docker containers can communicate via service name on shared networks
            if (self::is_docker_container()) {
                // Use service name for container-to-container communication
                // This works when both containers are on the same Docker network
                // The BrightTally service name is 'laravel.test' in docker-compose.yml
                if (defined('BRIGHT_TALLY_CONTAINER_URL')) {
                    return BRIGHT_TALLY_CONTAINER_URL;
                }
                // Try using the service name from BrightTally docker-compose
                // Service name is 'laravel.test' in docker-compose.yml
                // Use port 80 (container port, not host port)
                return 'http://laravel.test';
            }
            
            // For Docker Desktop on Mac, try host.docker.internal
            // This allows WordPress in Docker to access services on the Mac host
            if (isset($_SERVER['HTTP_HOST']) && (
                strpos($_SERVER['HTTP_HOST'], 'localhost:') !== false ||
                strpos($_SERVER['HTTP_HOST'], '.docker') !== false
            )) {
                // WordPress is likely in Docker, use host.docker.internal
                // But try service name first if available
                return 'http://laravel.test';
            }
            
            // Default local development URL
            return 'http://localhost:8000'; // Common Laravel default
        }
        
        return false;
    }
    
    /**
     * Check if WordPress is running in a Docker container
     *
     * @since 1.0.0
     * @return bool True if in Docker container
     */
    private static function is_docker_container() {
        // Check for Docker environment indicators
        // Cache the result to avoid multiple file checks
        static $is_docker = null;
        if ($is_docker !== null) {
            return $is_docker;
        }
        
        $is_docker = (
            // Check for .dockerenv file (most reliable)
            @file_exists('/.dockerenv') ||
            // Docker container name in hostname
            (isset($_SERVER['HOSTNAME']) && (
                strpos($_SERVER['HOSTNAME'], 'docker') !== false ||
                strpos($_SERVER['HOSTNAME'], 'brighttally') !== false ||
                strpos($_SERVER['HOSTNAME'], 'brighttally-wp') !== false
            )) ||
            // Check cgroup for Docker (if file is readable)
            (@file_exists('/proc/self/cgroup') && 
             ($cgroup = @file_get_contents('/proc/self/cgroup')) && 
             strpos($cgroup, 'docker') !== false) ||
            // Check if running in a container environment
            (isset($_SERVER['container']) && $_SERVER['container'] === 'docker')
        );
        
        return $is_docker;
    }
    
    /**
     * Check if we're in development mode
     *
     * @since 1.0.0
     * @return bool True if in development
     */
    public static function is_development() {
        $api_url = self::get_api_url();
        return (
            strpos($api_url, 'localhost') !== false ||
            strpos($api_url, '127.0.0.1') !== false ||
            strpos($api_url, '.local') !== false ||
            strpos($api_url, '.test') !== false ||
            strpos($api_url, 'http://') !== false // Development usually uses http
        );
    }
    
    /**
     * Get the current environment name
     *
     * @since 1.0.0
     * @return string Environment name (development, staging, production)
     */
    public static function get_environment() {
        if (self::is_development()) {
            return 'development';
        }
        
        $url = self::get_api_url();
        if (strpos($url, 'staging') !== false || strpos($url, 'stage') !== false) {
            return 'staging';
        }
        
        return 'production';
    }
    
    /**
     * Convert API URL to browser-accessible URL
     * 
     * This converts URLs like host.docker.internal:8000 or laravel.test to browser-accessible URLs
     * like brighttally.test or localhost:8000
     *
     * @since 1.0.0
     * @param string $url The URL to convert
     * @return string Browser-accessible URL
     */
    public static function to_browser_url($url) {
        if (empty($url)) {
            return $url;
        }
        
        // Check for BRIGHT_TALLY_BROWSER_URL constant (for Valet, etc.)
        if (defined('BRIGHT_TALLY_BROWSER_URL') && !empty(BRIGHT_TALLY_BROWSER_URL)) {
            // If URL contains the API URL, replace with browser URL
            $api_url = self::get_api_url();
            if (strpos($url, $api_url) === 0) {
                // Replace API URL base with browser URL base
                $path = substr($url, strlen($api_url));
                return rtrim(BRIGHT_TALLY_BROWSER_URL, '/') . $path;
            }
            
            // Also check if URL contains container service names or internal URLs
            // and replace with browser URL
            $parsed = parse_url($url);
            if ($parsed) {
                $host = $parsed['host'] ?? '';
                // Check for container service names or internal Docker URLs
                if ($host === 'host.docker.internal' || 
                    $host === 'laravel.test' ||
                    ($host === 'localhost' && isset($parsed['port']) && $parsed['port'] == 8000)) {
                    $browser_parsed = parse_url(BRIGHT_TALLY_BROWSER_URL);
                    if ($browser_parsed) {
                        $url = ($browser_parsed['scheme'] ?? 'http') . '://' . 
                               ($browser_parsed['host'] ?? 'localhost') . 
                               (isset($browser_parsed['port']) ? ':' . $browser_parsed['port'] : '') .
                               ($parsed['path'] ?? '') .
                               (isset($parsed['query']) ? '?' . $parsed['query'] : '') .
                               (isset($parsed['fragment']) ? '#' . $parsed['fragment'] : '');
                        return $url;
                    }
                }
            }
        }
        
        // Convert container service names to browser-accessible URLs
        // Default: brighttally.test for Valet/Laravel Sail
        $parsed = parse_url($url);
        if ($parsed && isset($parsed['host'])) {
            $host = $parsed['host'];
            
            // Convert laravel.test (container service) to brighttally.test (browser domain)
            if ($host === 'laravel.test') {
                $url = str_replace('laravel.test', 'brighttally.test', $url);
                // Remove port if it was on 80 (default HTTP)
                $url = preg_replace('/:80(\/|$)/', '$1', $url);
            }
            
            // Convert host.docker.internal to localhost or brighttally.test
            if ($host === 'host.docker.internal') {
                // Use brighttally.test if available, otherwise localhost
                $replacement = 'brighttally.test';
                $url = str_replace('host.docker.internal', $replacement, $url);
                // Remove port if it was on 80
                $url = preg_replace('/:80(\/|$)/', '$1', $url);
            }
        }
        
        return $url;
    }

    /**
     * Get the documentation base URL
     *
     * Priority:
     * 1. WordPress option override
     * 2. BRIGHT_TALLY_DOCS_URL constant
     * 3. Derived from API URL with /docs suffix
     *
     * @since 1.1.0
     * @return string Documentation base URL (without trailing slash)
     */
    public static function get_docs_url() {
        $saved_url = get_option('bright_tally_docs_url', '');
        if (!empty($saved_url)) {
            return rtrim($saved_url, '/');
        }

        if (defined('BRIGHT_TALLY_DOCS_URL') && !empty(BRIGHT_TALLY_DOCS_URL)) {
            return rtrim(BRIGHT_TALLY_DOCS_URL, '/');
        }

        return rtrim(self::get_api_url(), '/') . '/docs';
    }
}

