<?php
/**
 * BrightTally Plugin Updater
 *
 * Handles version updates and database migrations
 *
 * @package Bright_Tally
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * BrightTally Plugin Updater Class
 *
 * @since 1.0.0
 */
class Bright_Tally_Updater {
    
    /**
     * Static cache to prevent multiple checks in the same request
     *
     * @since 1.0.0
     * @var bool
     */
    private static $checked = false;
    
    /**
     * Check and run updates if needed
     *
     * Uses WordPress's built-in option caching and static variable
     * to ensure we only check once per request.
     *
     * @since 1.0.0
     */
    public static function check_updates() {
        // Only check once per request
        if (self::$checked) {
            return;
        }
        
        self::$checked = true;
        
        // WordPress caches options in memory after first get_option() call
        // This is very efficient and doesn't hit the database on subsequent calls
        $installed_version = get_option('bright_tally_version', '0.0.0');
        $current_version = BRIGHT_TALLY_VERSION;
        
        // Only run updates if version has changed
        if (version_compare($installed_version, $current_version, '<')) {
            $result = self::run_updates($installed_version, $current_version);
            
            // Only update version if update succeeded
            if (!is_wp_error($result)) {
                update_option('bright_tally_version', $current_version);
                // Store success message
                $success_message = sprintf(
                    esc_html__('Plugin updated from version %s to %s', 'bright-tally'),
                    $installed_version,
                    $current_version
                );
                set_transient('bright_tally_update_success', $success_message, 300);
            } else {
                // Store error for admin notice
                $error_message = $result->get_error_message();
                set_transient('bright_tally_update_error', $error_message, 300);
            }
            
            // Clear static cache after update to allow re-checking if needed
            self::$checked = false;
        }
    }
    
    /**
     * Run version-specific updates
     *
     * @since 1.0.0
     * @param string $from_version The version we're updating from
     * @param string $to_version The version we're updating to
     * @return bool|WP_Error True on success, WP_Error on failure
     */
    private static function run_updates($from_version, $to_version) {
        // Allow plugins to hook into update process
        do_action('bright_tally_before_update', $from_version, $to_version);
        
        // Log the update
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log(sprintf(
                'BrightTally: Updating from version %s to %s',
                $from_version,
                $to_version
            ));
        }
        
        $errors = array();
        
        // Run version-specific update methods
        try {
            if (version_compare($from_version, '1.0.0', '<')) {
                $result = self::update_to_1_0_0();
                if (is_wp_error($result)) {
                    $errors[] = $result->get_error_message();
                }
            }
            
            // Add future update methods here:
            // if (version_compare($from_version, '1.1.0', '<')) {
            //     $result = self::update_to_1_1_0();
            //     if (is_wp_error($result)) {
            //         $errors[] = $result->get_error_message();
            //     }
            // }
            
            // If there were errors, store them and show admin notice
            if (!empty($errors)) {
                $error_message = implode(' ', $errors);
                set_transient('bright_tally_update_error', $error_message, 300); // 5 minutes
                
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    error_log('BrightTally: Update failed with errors: ' . $error_message);
                }
                
                // Still clear caches and run after hook, but return error
                self::clear_caches();
                do_action('bright_tally_after_update', $from_version, $to_version);
                
                return new WP_Error('update_failed', $error_message);
            }
            
            // Clear any caches
            self::clear_caches();
            
            // Allow plugins to hook into update completion
            do_action('bright_tally_after_update', $from_version, $to_version);
            
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('BrightTally: Update completed successfully');
            }
            
            return true;
            
        } catch (Exception $e) {
            $error_message = sprintf(
                esc_html__('Update failed: %s', 'bright-tally'),
                $e->getMessage()
            );
            set_transient('bright_tally_update_error', $error_message, 300);
            
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('BrightTally: Update exception: ' . $e->getMessage());
            }
            
            return new WP_Error('update_exception', $error_message);
        }
    }
    
    /**
     * Update to version 1.0.0
     *
     * @since 1.0.0
     * @return bool|WP_Error True on success, WP_Error on failure
     */
    private static function update_to_1_0_0() {
        // This is the initial version, so we just ensure everything is set up
        // Future versions can add migration logic here
        
        // Ensure database tables exist
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();
        $table_name = $wpdb->prefix . 'bright_tally_polls';
        
        // Check if table exists (sanitize table name first)
        $table_name_safe = preg_replace('/[^a-zA-Z0-9_]/', '', $table_name);
        
        // Use a safer method to check table existence
        $table_exists = $wpdb->get_var($wpdb->prepare(
            "SHOW TABLES LIKE %s",
            $table_name_safe
        ));
        
        if (!$table_exists) {
            require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
            
            $sql = "CREATE TABLE $table_name (
                id mediumint(9) NOT NULL AUTO_INCREMENT,
                poll_id varchar(255) NOT NULL,
                poll_title text NOT NULL,
                poll_data longtext NOT NULL,
                created_at datetime DEFAULT CURRENT_TIMESTAMP,
                updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (id),
                UNIQUE KEY poll_id (poll_id)
            ) $charset_collate;";
            
            dbDelta($sql);
            
            // Check if table was actually created
            $table_exists_after = $wpdb->get_var($wpdb->prepare(
                "SHOW TABLES LIKE %s",
                $table_name_safe
            ));
            
            if (!$table_exists_after) {
                return new WP_Error(
                    'table_creation_failed',
                    sprintf(
                        esc_html__('Failed to create database table: %s', 'bright-tally'),
                        $table_name
                    )
                );
            }
        }
        
        // Ensure default options are set
        $default_options = array(
            'bright_tally_api_key' => '',
            'bright_tally_api_url' => 'https://brighttally.com',
            'bright_tally_default_theme' => 'light',
            'bright_tally_cache_duration' => 300,
        );
        
        foreach ($default_options as $option_name => $option_value) {
            if (get_option($option_name) === false) {
                $result = add_option($option_name, $option_value);
                if (!$result) {
                    return new WP_Error(
                        'option_creation_failed',
                        sprintf(
                            esc_html__('Failed to create option: %s', 'bright-tally'),
                            $option_name
                        )
                    );
                }
            }
        }
        
        return true;
    }
    
    /**
     * Clear plugin caches
     *
     * @since 1.0.0
     */
    private static function clear_caches() {
        // Clear transients
        delete_transient('bright_tally_polls_cache');
        delete_transient('bright_tally_api_connection_test');
        
        // Clear object cache if available
        if (function_exists('wp_cache_flush')) {
            wp_cache_flush();
        }
    }
    
    /**
     * Get the installed version
     *
     * @since 1.0.0
     * @return string The installed version
     */
    public static function get_installed_version() {
        return get_option('bright_tally_version', '0.0.0');
    }
    
    /**
     * Get the current version
     *
     * @since 1.0.0
     * @return string The current version
     */
    public static function get_current_version() {
        return BRIGHT_TALLY_VERSION;
    }
    
    /**
     * Display admin notices for update errors
     *
     * @since 1.0.0
     */
    public static function display_update_notices() {
        // Only show on admin pages
        if (!is_admin()) {
            return;
        }
        
        // Check for update errors
        $update_error = get_transient('bright_tally_update_error');
        if ($update_error) {
            ?>
            <div class="notice notice-error is-dismissible">
                <p>
                    <strong><?php esc_html_e('BrightTally Update Error', 'bright-tally'); ?>:</strong>
                    <?php echo esc_html($update_error); ?>
                </p>
                <p>
                    <?php esc_html_e('The plugin may not function correctly. Please contact support if this issue persists.', 'bright-tally'); ?>
                </p>
            </div>
            <?php
            // Clear the transient after displaying
            delete_transient('bright_tally_update_error');
        }
        
        // Check for successful updates
        $update_success = get_transient('bright_tally_update_success');
        if ($update_success) {
            ?>
            <div class="notice notice-success is-dismissible">
                <p>
                    <strong><?php esc_html_e('BrightTally Updated Successfully', 'bright-tally'); ?>:</strong>
                    <?php echo esc_html($update_success); ?>
                </p>
            </div>
            <?php
            // Clear the transient after displaying
            delete_transient('bright_tally_update_success');
        }
    }
}

