<?php
/**
 * Token manager for secure storage and retrieval of API tokens
 *
 * @package Bright_Tally
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Bright_Tally_Token_Manager {
    
    /**
     * Store encrypted access token
     *
     * @since 1.0.0
     * @param string $token The access token
     * @return bool True on success, false on failure
     */
    public static function store_access_token($token) {
        $encrypted = self::encrypt($token);
        return update_option('bright_tally_access_token', $encrypted, false);
    }
    
    /**
     * Retrieve and decrypt access token
     *
     * @since 1.0.0
     * @return string|false The access token or false on failure
     */
    public static function get_access_token() {
        $encrypted = get_option('bright_tally_access_token', false);
        if (!$encrypted) {
            return false;
        }
        return self::decrypt($encrypted);
    }
    
    /**
     * Delete access token
     *
     * @since 1.0.0
     * @return bool True on success, false on failure
     */
    public static function delete_access_token() {
        return delete_option('bright_tally_access_token');
    }
    
    /**
     * Store token expiration timestamp
     *
     * @since 1.0.0
     * @param int $timestamp Unix timestamp
     * @return bool True on success, false on failure
     */
    public static function store_token_expires_at($timestamp) {
        return update_option('bright_tally_token_expires_at', $timestamp, false);
    }
    
    /**
     * Get token expiration timestamp
     *
     * @since 1.0.0
     * @return int|false Unix timestamp or false
     */
    public static function get_token_expires_at() {
        return get_option('bright_tally_token_expires_at', false);
    }
    
    /**
     * Check if token is expired
     *
     * @since 1.0.0
     * @return bool True if expired or missing, false if valid
     */
    public static function is_token_expired() {
        $expires_at = self::get_token_expires_at();
        if (!$expires_at) {
            return true; // No expiration set, assume expired
        }
        return time() >= $expires_at;
    }
    
    /**
     * Store user account information
     *
     * @since 1.0.0
     * @param array $user_data User data array
     * @return bool True on success, false on failure
     */
    public static function store_user_data($user_data) {
        $encrypted = self::encrypt(wp_json_encode($user_data));
        return update_option('bright_tally_user_data', $encrypted, false);
    }
    
    /**
     * Get user account information
     *
     * @since 1.0.0
     * @return array|false User data array or false on failure
     */
    public static function get_user_data() {
        $encrypted = get_option('bright_tally_user_data', false);
        if (!$encrypted) {
            return false;
        }
        $decrypted = self::decrypt($encrypted);
        return json_decode($decrypted, true);
    }
    
    /**
     * Store connection status
     *
     * @since 1.0.0
     * @param string $status Connection status (connected, disconnected, error)
     * @return bool True on success, false on failure
     */
    public static function store_connection_status($status) {
        return update_option('bright_tally_connection_status', $status, false);
    }
    
    /**
     * Get connection status
     *
     * @since 1.0.0
     * @return string Connection status
     */
    public static function get_connection_status() {
        return get_option('bright_tally_connection_status', 'disconnected');
    }
    
    /**
     * Check if account is connected
     *
     * @since 1.0.0
     * @return bool True if connected, false otherwise
     */
    public static function is_connected() {
        $token = self::get_access_token();
        $status = self::get_connection_status();
        return !empty($token) && $status === 'connected' && !self::is_token_expired();
    }
    
    /**
     * Clear all stored tokens and user data
     *
     * @since 1.0.0
     * @return void
     */
    public static function clear_all() {
        self::delete_access_token();
        delete_option('bright_tally_token_expires_at');
        delete_option('bright_tally_user_data');
        self::store_connection_status('disconnected');
    }
    
    /**
     * Encrypt a string using WordPress salts
     *
     * @since 1.0.0
     * @param string $data Data to encrypt
     * @return string Encrypted data
     */
    private static function encrypt($data) {
        if (!function_exists('openssl_encrypt')) {
            // Fallback to base64 encoding if OpenSSL is not available
            return base64_encode($data);
        }
        
        $method = 'AES-256-CBC';
        $key = self::get_encryption_key();
        $iv = openssl_random_pseudo_bytes(openssl_cipher_iv_length($method));
        
        $encrypted = openssl_encrypt($data, $method, $key, 0, $iv);
        return base64_encode($encrypted . '::' . $iv);
    }
    
    /**
     * Decrypt a string using WordPress salts
     *
     * @since 1.0.0
     * @param string $data Encrypted data
     * @return string|false Decrypted data or false on failure
     */
    private static function decrypt($data) {
        if (!function_exists('openssl_decrypt')) {
            // Fallback to base64 decoding if OpenSSL is not available
            return base64_decode($data);
        }
        
        $method = 'AES-256-CBC';
        $key = self::get_encryption_key();
        
        $data = base64_decode($data);
        list($encrypted_data, $iv) = explode('::', $data, 2);
        
        return openssl_decrypt($encrypted_data, $method, $key, 0, $iv);
    }
    
    /**
     * Get encryption key from WordPress salts
     *
     * @since 1.0.0
     * @return string Encryption key
     */
    private static function get_encryption_key() {
        // Use WordPress salts for encryption key
        $key = AUTH_SALT . AUTH_KEY . NONCE_SALT . SECURE_AUTH_SALT;
        return hash('sha256', $key . get_option('siteurl'));
    }
}

