<?php
/**
 * Shortcode functionality for displaying polls
 *
 * @package Bright_Tally
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Bright_Tally_Shortcode {
    
    /**
     * Register the shortcode
     *
     * @since 1.0.0
     */
    public function register_shortcode() {
        add_shortcode('bright_tally', array($this, 'render_shortcode'));
        add_shortcode('brighttally', array($this, 'render_shortcode')); // Alternative shortcode
    }
    
    /**
     * Render the shortcode
     *
     * @since 1.0.0
     * @param array $atts Shortcode attributes
     * @return string The shortcode output
     */
    public function render_shortcode($atts) {
        // Allow filtering of default attributes
        $default_atts = apply_filters('bright_tally_shortcode_defaults', array(
            'id' => '',
            'theme' => get_option('bright_tally_default_theme', 'light'),
            'width' => '100%',
            'height' => 'auto',
            'show_results' => 'false',
            'allow_voting' => 'true'
        ));
        
        $atts = shortcode_atts($default_atts, $atts, 'bright_tally');
        
        // Allow filtering of parsed attributes
        $atts = apply_filters('bright_tally_shortcode_atts', $atts);
        
        if (empty($atts['id'])) {
            return '<div class="bright-tally-error">' . esc_html__('Error: Poll ID is required.', 'bright-tally') . '</div>';
        }
        
        // Validate poll ID format to prevent path traversal and XSS
        $poll_id = sanitize_text_field($atts['id']);
        
        // Validate poll ID length
        if (strlen($poll_id) > 255) {
            return '<div class="bright-tally-error">' . esc_html__('Error: Poll ID is too long.', 'bright-tally') . '</div>';
        }
        
        // Validate poll ID is not empty after sanitization
        if (empty($poll_id)) {
            return '<div class="bright-tally-error">' . esc_html__('Error: Poll ID is required.', 'bright-tally') . '</div>';
        }
        
        if (!preg_match('/^[a-zA-Z0-9_-]+$/', $poll_id)) {
            return '<div class="bright-tally-error">' . esc_html__('Error: Invalid poll ID format.', 'bright-tally') . '</div>';
        }
        
        // Validate and sanitize theme
        $allowed_themes = array('light', 'dark');
        $theme_raw = isset($atts['theme']) ? sanitize_text_field($atts['theme']) : 'light';
        $theme = in_array($theme_raw, $allowed_themes, true) ? $theme_raw : 'light';
        
        // Validate width (must be CSS-safe)
        $width_raw = isset($atts['width']) ? sanitize_text_field($atts['width']) : '100%';
        $width = '100%'; // Default
        
        // Validate width format (percentage, pixels, or auto)
        if (preg_match('/^(\d+%|\d+px|auto)$/', $width_raw)) {
            $width = $width_raw;
        } elseif (preg_match('/^\d+$/', $width_raw)) {
            // If just a number, assume pixels
            $width = absint($width_raw) . 'px';
            // Limit max width to prevent layout issues
            if (absint($width_raw) > 10000) {
                $width = '100%';
            }
        }
        
        // Validate height (must be CSS-safe or 'auto')
        $height_raw = isset($atts['height']) ? sanitize_text_field($atts['height']) : 'auto';
        $height = 'auto'; // Default
        
        if ($height_raw === 'auto') {
            $height = 'auto';
        } elseif (preg_match('/^\d+px$/', $height_raw)) {
            // Validate pixel height is reasonable (max 10000px)
            $pixel_value = absint(str_replace('px', '', $height_raw));
            if ($pixel_value > 0 && $pixel_value <= 10000) {
                $height = $pixel_value . 'px';
            }
        } elseif (preg_match('/^\d+$/', $height_raw)) {
            // If just a number, assume pixels
            $pixel_value = absint($height_raw);
            if ($pixel_value > 0 && $pixel_value <= 10000) {
                $height = $pixel_value . 'px';
            }
        }
        
        // Validate boolean attributes
        $show_results = isset($atts['show_results']) ? filter_var($atts['show_results'], FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE) : false;
        $allow_voting = isset($atts['allow_voting']) ? filter_var($atts['allow_voting'], FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE) : true;
        
        // If filter_var returns null, use string comparison as fallback
        if ($show_results === null) {
            $show_results = in_array(strtolower($atts['show_results'] ?? ''), array('true', '1', 'yes'), true);
        }
        if ($allow_voting === null) {
            $allow_voting = !in_array(strtolower($atts['allow_voting'] ?? ''), array('false', '0', 'no'), true);
        }
        
        // Get API URL (allow filtering)
        $api_url = apply_filters('bright_tally_api_url', Bright_Tally_URL_Helper::get_api_url());
        $embed_url = rtrim($api_url, '/') . '/embed/' . urlencode($poll_id);
        
        // Allow filtering of embed URL
        $embed_url = apply_filters('bright_tally_embed_url', $embed_url, $poll_id, $atts);
        
        // Generate unique ID for this poll instance
        $instance_id = 'bright-tally-' . sanitize_html_class($poll_id) . '-' . uniqid();
        
        // Calculate iframe height
        $iframe_height = ($height === 'auto') ? '600' : $height;
        
        // Start output buffering
        ob_start();
        ?>
        <div id="<?php echo esc_attr($instance_id); ?>" 
             class="bright-tally-poll-wrapper" 
             style="width: <?php echo esc_attr($width); ?>; max-width: 100%;">
            
            <div class="bright-tally-loading" style="display: none;">
                <div class="bright-tally-spinner"></div>
                <p><?php esc_html_e('Loading poll...', 'bright-tally'); ?></p>
            </div>
            
            <iframe 
                id="<?php echo esc_attr($instance_id); ?>-iframe"
                src="<?php echo esc_url($embed_url); ?>"
                width="100%"
                height="<?php echo esc_attr($iframe_height); ?>"
                frameborder="0"
                scrolling="no"
                style="border: none; display: block; min-height: 400px;"
                title="<?php echo esc_attr(__('BrightTally Poll', 'bright-tally')); ?>"
                loading="lazy"
            ></iframe>
            
            <div class="bright-tally-error" style="display: none;">
                <p><?php esc_html_e('Sorry, there was an error loading this poll.', 'bright-tally'); ?></p>
            </div>
        </div>
        
        <script>
        (function() {
            var iframe = document.getElementById('<?php echo esc_js($instance_id); ?>-iframe');
            var wrapper = document.getElementById('<?php echo esc_js($instance_id); ?>');
            var loading = wrapper.querySelector('.bright-tally-loading');
            var error = wrapper.querySelector('.bright-tally-error');
            
            // Show loading initially
            if (loading) {
                loading.style.display = 'block';
            }
            
            // Handle iframe load
            iframe.addEventListener('load', function() {
                if (loading) {
                    loading.style.display = 'none';
                }
                
                // Listen for resize messages from iframe
                window.addEventListener('message', function(event) {
                    // Verify origin to prevent XSS
                    var apiUrl = '<?php echo esc_js($api_url); ?>';
                    var allowedOrigins = [
                        apiUrl,
                        'https://brighttally.com',
                        'https://www.brighttally.com'
                    ];
                    
                    // Parse origin to compare properly
                    try {
                        var originUrl = new URL(event.origin);
                        var apiUrlObj = new URL(apiUrl);
                        var isAllowed = false;
                        
                        for (var i = 0; i < allowedOrigins.length; i++) {
                            try {
                                var allowedUrl = new URL(allowedOrigins[i]);
                                if (originUrl.origin === allowedUrl.origin) {
                                    isAllowed = true;
                                    break;
                                }
                            } catch (e) {
                                // Invalid URL, skip
                            }
                        }
                        
                        if (!isAllowed) {
                            return;
                        }
                    } catch (e) {
                        // Invalid origin URL, reject
                        return;
                    }
                    
                    // Handle resize message
                    if (event.data && event.data.type === 'resize' && event.data.height) {
                        iframe.style.height = event.data.height + 'px';
                    }
                });
            });
            
            // Handle iframe error
            iframe.addEventListener('error', function() {
                if (loading) {
                    loading.style.display = 'none';
                }
                if (error) {
                    error.style.display = 'block';
                }
                iframe.style.display = 'none';
            });
            
            // Auto-resize on load (fallback)
            setTimeout(function() {
                try {
                    var iframeDoc = iframe.contentDocument || iframe.contentWindow.document;
                    if (iframeDoc && iframeDoc.body) {
                        var height = iframeDoc.body.scrollHeight;
                        if (height > 0) {
                            iframe.style.height = height + 'px';
                        }
                    }
                } catch (e) {
                    // Cross-origin error is expected, ignore
                }
            }, 1000);
        })();
        </script>
        <?php
        
        $output = ob_get_clean();
        
        // Allow filtering of final shortcode output
        return apply_filters('bright_tally_shortcode_output', $output, $poll_id, $atts);
    }
}
