<?php
/**
 * OAuth handler for BrightTally WordPress plugin
 *
 * @package Bright_Tally
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Bright_Tally_OAuth {
    
    /**
     * The API base URL
     *
     * @since 1.0.0
     * @var string
     */
    private $api_url;
    
    /**
     * Initialize the OAuth class
     *
     * @since 1.0.0
     */
    public function __construct() {
        // Use URL helper for environment-aware URL detection
        $this->api_url = Bright_Tally_URL_Helper::get_api_url();
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('BrightTally: API URL=' . $this->api_url);
        }
    }
    
    /**
     * Get OAuth authorization URL
     *
     * @since 1.0.0
     * @param string $redirect_uri The callback URL
     * @param string $state Optional state parameter for CSRF protection
     * @return string|WP_Error Authorization URL or error
     */
    public function get_authorization_url($redirect_uri, $state = '') {
        $url = rtrim($this->api_url, '/') . '/api/wordpress/oauth/authorize';
        
        $args = array(
            'method' => 'POST',
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ),
            'body' => wp_json_encode(array(
                'redirect_uri' => $redirect_uri,
                'state' => $state,
            )),
        );
        
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('BrightTally: Request URL=' . $url . ' with args=' . print_r($args, true));
        }
        $response = wp_remote_request($url, $args);
        
        if (is_wp_error($response)) {
            // Add more context to connection errors
            $error_message = $response->get_error_message();
            if (strpos($error_message, 'Could not resolve host') !== false || 
                strpos($error_message, 'Connection refused') !== false ||
                strpos($error_message, 'timed out') !== false) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    error_log('BrightTally: Connection error=' . $error_message);
                }
                return new WP_Error(
                    'connection_error',
                    'Cannot connect to BrightTally API. Please check your API URL setting and ensure the server is running.',
                    array('original_error' => $error_message)
                );
            }
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('BrightTally: Connection error=' . $error_message);
            }
            return $response;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);
        
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('BrightTally: Response=' . print_r($response, true));
            error_log('BrightTally: Data=' . print_r($data, true));
        }
        // Log response for debugging (always log in development, or if response code is not 200)
        $should_log = (defined('WP_DEBUG') && WP_DEBUG) || $response_code >= 400;
        if ($should_log) {
            error_log('BrightTally OAuth Response: Code=' . $response_code . ', Body=' . $response_body);
            error_log('BrightTally OAuth Parsed Data: ' . print_r($data, true));
            error_log('BrightTally: has requires_login? ' . (isset($data['requires_login']) ? 'YES' : 'NO'));
            if (isset($data['requires_login'])) {
                error_log('BrightTally: requires_login value=' . ($data['requires_login'] ? 'true' : 'false'));
            }
        }
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('BrightTally: Response code=' . $response_code);
        }
        // Check if login is required FIRST (even for 401 status)
        // This should be checked before treating 400+ as errors
        if (isset($data['requires_login']) && $data['requires_login']) {
            $login_url = $data['login_url'] ?? '';
            
            // Convert API URL to browser-accessible URL
            // This handles cases like host.docker.internal:8000 -> brighttally.test
            // or localhost:8000 -> brighttally.test (for Valet)
            if (!empty($login_url)) {
                $login_url = Bright_Tally_URL_Helper::to_browser_url($login_url);
            }
            
            // Already checked $should_log above
            return new WP_Error(
                'requires_login',
                $data['message'] ?? 'Login required',
                array(
                    'login_url' => $login_url,
                )
            );
        }
        
        // Now check for other errors (but not requires_login)
        if ($response_code >= 400) {
            $error_msg = isset($data['error_description']) ? $data['error_description'] : 'OAuth authorization failed';
            if ($should_log) {
                error_log('BrightTally: OAuth error - Code=' . $response_code . ', Message=' . $error_msg);
                error_log('BrightTally: Full response data: ' . print_r($data, true));
                error_log('BrightTally: NOTE - requires_login was NOT detected in response!');
            }
            return new WP_Error(
                'oauth_error',
                $error_msg,
                array('status' => $response_code, 'response_data' => $data)
            );
        }
        
        // Return auth_url (which is the callback URL with code) for popup
        return $data['auth_url'] ?? $data['callback_url'] ?? '';
    }
    
    /**
     * Exchange authorization code for access token
     *
     * @since 1.0.0
     * @param string $code Authorization code
     * @param string $redirect_uri The callback URL
     * @return array|WP_Error Token data or error
     */
    public function exchange_code_for_token($code, $redirect_uri) {
        $url = rtrim($this->api_url, '/') . '/api/wordpress/oauth/callback';
        
        $args = array(
            'method' => 'POST',
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ),
            'body' => wp_json_encode(array(
                'code' => $code,
                'redirect_uri' => $redirect_uri,
            )),
        );
        
        $response = wp_remote_request($url, $args);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);
        
        if ($response_code >= 400) {
            return new WP_Error(
                'oauth_error',
                isset($data['error_description']) ? $data['error_description'] : 'Token exchange failed',
                array('status' => $response_code)
            );
        }
        
        return $data;
    }
    
    /**
     * Refresh access token
     *
     * @since 1.0.0
     * @return array|WP_Error New token data or error
     */
    public function refresh_token() {
        $token_manager = new Bright_Tally_Token_Manager();
        $current_token = $token_manager::get_access_token();
        
        if (!$current_token) {
            return new WP_Error('no_token', 'No access token available to refresh');
        }
        
        $url = rtrim($this->api_url, '/') . '/api/wordpress/oauth/refresh';
        
        $args = array(
            'method' => 'POST',
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'Authorization' => 'Bearer ' . $current_token,
            ),
            'body' => wp_json_encode(array(
                'refresh_token' => $current_token, // Using current token for simplicity
            )),
        );
        
        $response = wp_remote_request($url, $args);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);
        
        if ($response_code >= 400) {
            return new WP_Error(
                'refresh_error',
                isset($data['error_description']) ? $data['error_description'] : 'Token refresh failed',
                array('status' => $response_code)
            );
        }
        
        return $data;
    }
    
    /**
     * Revoke access token
     *
     * @since 1.0.0
     * @return bool|WP_Error True on success or error
     */
    public function revoke_token() {
        $token_manager = new Bright_Tally_Token_Manager();
        $current_token = $token_manager::get_access_token();
        
        if (!$current_token) {
            return true; // Already revoked
        }
        
        $url = rtrim($this->api_url, '/') . '/api/wordpress/oauth/revoke';
        
        $args = array(
            'method' => 'POST',
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'Authorization' => 'Bearer ' . $current_token,
            ),
        );
        
        $response = wp_remote_request($url, $args);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        
        if ($response_code >= 400) {
            return new WP_Error('revoke_error', 'Token revocation failed', array('status' => $response_code));
        }
        
        // Clear local tokens
        $token_manager::clear_all();
        
        return true;
    }
    
    /**
     * Login with email and password
     *
     * @since 1.0.0
     * @param string $email User email
     * @param string $password User password
     * @return array|WP_Error Token data or error
     */
    public function login($email, $password) {
        $url = rtrim($this->api_url, '/') . '/api/wordpress/auth/login';
        
        $args = array(
            'method' => 'POST',
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ),
            'body' => wp_json_encode(array(
                'email' => $email,
                'password' => $password,
            )),
        );
        
        $response = wp_remote_request($url, $args);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);
        
        if ($response_code >= 400) {
            return new WP_Error(
                'login_error',
                isset($data['error_description']) 
                    ? $data['error_description'] 
                    : (isset($data['message']) ? $data['message'] : 'Login failed. Please check your credentials.'),
                array('status' => $response_code)
            );
        }
        
        return $data;
    }
    
    /**
     * Create account from WordPress plugin
     *
     * @since 1.0.0
     * @param array $user_data User data (name, email, password)
     * @return array|WP_Error Token data or error
     */
    public function create_account($user_data) {
        $url = rtrim($this->api_url, '/') . '/api/wordpress/account/create';
        
        $args = array(
            'method' => 'POST',
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ),
            'body' => wp_json_encode($user_data),
        );
        
        $response = wp_remote_request($url, $args);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);
        
        if ($response_code >= 400) {
            return new WP_Error(
                'account_creation_error',
                isset($data['error_description']) 
                    ? $data['error_description'] 
                    : (isset($data['errors']) ? wp_json_encode($data['errors']) : 'Account creation failed'),
                array('status' => $response_code)
            );
        }
        
        return $data;
    }
}

