<?php
/**
 * API functionality for communicating with BrightTally service
 *
 * @package Bright_Tally
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Bright_Tally_API {
    
    /**
     * The API base URL
     *
     * @since 1.0.0
     * @var string
     */
    private $api_url;
    
    /**
     * The API key (for backward compatibility)
     *
     * @since 1.0.0
     * @var string
     */
    private $api_key;
    
    /**
     * Token manager instance
     *
     * @since 1.0.0
     * @var Bright_Tally_Token_Manager
     */
    private $token_manager;
    
    /**
     * Initialize the API class
     *
     * @since 1.0.0
     */
    public function __construct() {
        $this->api_url = Bright_Tally_URL_Helper::get_api_url();
        $this->api_key = get_option('bright_tally_api_key', '');
        $this->token_manager = new Bright_Tally_Token_Manager();

        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('DEBUG: api_url: ' . print_r($this->api_url, true));
            error_log('DEBUG: api_key: ' . print_r($this->api_key, true));
        }
    }
    
    /**
     * Get authentication token (prefer OAuth token, fallback to API key)
     *
     * @since 1.0.0
     * @return string|false Token or false if not available
     */
    private function get_auth_token() {
        // Try OAuth token first
        $token = Bright_Tally_Token_Manager::get_access_token();
        if ($token && !Bright_Tally_Token_Manager::is_token_expired()) {
            return $token;
        }
        
        // Fallback to API key for backward compatibility
        if (!empty($this->api_key)) {
            return $this->api_key;
        }
        
        return false;
    }
    
    /**
     * Make a request to the BrightTally API
     *
     * @since 1.0.0
     * @param string $endpoint The API endpoint
     * @param array $data The data to send
     * @param string $method The HTTP method
     * @return array|WP_Error The response data or error
     */
    public function make_request($endpoint, $data = array(), $method = 'GET') {
        $token = $this->get_auth_token();
        
        if (!$token) {
            return new WP_Error('no_auth', 'No authentication token available. Please connect your account.');
        }
        
        $url = rtrim($this->api_url, '/') . '/api/' . ltrim($endpoint, '/');
        
        $args = array(
            'method' => $method,
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'Authorization' => 'Bearer ' . $token,
                'User-Agent' => 'BrightTally-WordPress-Plugin/' . BRIGHT_TALLY_VERSION
            )
        );
        
        if (!empty($data) && in_array($method, array('POST', 'PUT', 'PATCH'))) {
            $args['body'] = wp_json_encode($data);
        } elseif (!empty($data) && $method === 'GET') {
            $url .= '?' . http_build_query($data);
        }
        
        $response = wp_remote_request($url, $args);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        // Handle 401 Unauthorized - token might be expired
        if ($response_code === 401) {
            // Try to refresh token if using OAuth
            if (Bright_Tally_Token_Manager::get_access_token()) {
                $oauth = new Bright_Tally_OAuth();
                $refresh_result = $oauth->refresh_token();
                
                if (!is_wp_error($refresh_result) && isset($refresh_result['access_token'])) {
                    // Store new token and retry request
                    Bright_Tally_Token_Manager::store_access_token($refresh_result['access_token']);
                    if (isset($refresh_result['expires_in'])) {
                        Bright_Tally_Token_Manager::store_token_expires_at(time() + $refresh_result['expires_in']);
                    }
                    return $this->make_request($endpoint, $data, $method);
                }
            }
            
            return new WP_Error('unauthorized', 'Authentication failed. Please reconnect your account.', array('status' => 401));
        }
        
        if ($response_code >= 400) {
            $error_data = json_decode($response_body, true);
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('DEBUG: error_data: ' . print_r($error_data, true));
            }
            $error_message = isset($error_data['error_description']) 
                ? $error_data['error_description'] 
                : (isset($error_data['error']) ? $error_data['error'] : 'API request failed - Try reconnecting your account. If the problem persists, please contact support.');
            
            return new WP_Error('api_error', $error_message, array('status' => $response_code));
        }
        
        return json_decode($response_body, true);
    }
    
    /**
     * Get poll data from the API (public endpoint, no auth required)
     *
     * @since 1.0.0
     * @param string $poll_slug The poll slug or ID
     * @return array|WP_Error The poll data or error
     */
    public function get_poll($poll_slug) {
        // Validate poll slug format to prevent path traversal
        if (!preg_match('/^[a-zA-Z0-9_-]+$/', $poll_slug)) {
            return new WP_Error('invalid_poll_id', 'Invalid poll ID format');
        }
        
        // Use the public API endpoint that accepts slugs (no authentication required)
        $api_url = Bright_Tally_URL_Helper::get_api_url();
        $endpoint = rtrim($api_url, '/') . '/api/poll/' . urlencode($poll_slug);
        
        $args = array(
            'method' => 'GET',
            'timeout' => 30,
            'headers' => array(
                'Accept' => 'application/json',
                'User-Agent' => 'BrightTally-WordPress-Plugin/' . BRIGHT_TALLY_VERSION
            ),
        );

        $response = wp_remote_request($endpoint, $args);

        if (is_wp_error($response)) {
            return $response;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);

        if ($response_code >= 400) {
            $error_message = isset($data['error']) ? $data['error'] : 'Failed to load poll';
            return new WP_Error('api_error', $error_message, array('status' => $response_code));
        }

        return $data;
    }
    
    /**
     * Get all polls for the authenticated user
     *
     * @since 1.0.0
     * @param array $params Query parameters
     * @return array|WP_Error The polls data or error
     */
    public function get_polls($params = array()) {
        return $this->make_request('wordpress/polls', $params);
    }
    
    /**
     * Submit a poll response
     *
     * @since 1.0.0
     * @param string $poll_id The poll ID
     * @param array $response_data The response data
     * @return array|WP_Error The response or error
     */
    public function submit_poll_response($poll_id, $response_data) {
        // Validate poll ID format to prevent path traversal
        if (!preg_match('/^[a-zA-Z0-9_-]+$/', $poll_id)) {
            return new WP_Error('invalid_poll_id', 'Invalid poll ID format');
        }
        
        return $this->make_request('polls/' . urlencode($poll_id) . '/responses', $response_data, 'POST');
    }
    
    /**
     * Get poll results
     *
     * @since 1.0.0
     * @param string $poll_id The poll ID
     * @return array|WP_Error The poll results or error
     */
    public function get_poll_results($poll_id) {
        // Validate poll ID format to prevent path traversal
        if (!preg_match('/^[a-zA-Z0-9_-]+$/', $poll_id)) {
            return new WP_Error('invalid_poll_id', 'Invalid poll ID format');
        }
        
        return $this->make_request('polls/' . urlencode($poll_id) . '/results');
    }
    
    /**
     * Check if the API is properly configured
     *
     * @since 1.0.0
     * @return bool True if configured, false otherwise
     */
    public function is_configured() {
        // Check for OAuth token first (preferred)
        if (Bright_Tally_Token_Manager::is_connected()) {
            return true;
        }
        
        // Fallback to API key check for backward compatibility
        return !empty($this->api_key) && !empty($this->api_url);
    }
    
    /**
     * Test the API connection
     *
     * @since 1.0.0
     * @return array|WP_Error The test result or error
     */
    public function test_connection() {
        return $this->make_request('health');
    }
}
