<?php
/**
 * BrightTally Plugin Activator
 *
 * Handles plugin activation tasks including database table creation
 * and default option settings.
 *
 * @package BrightTally
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * BrightTally Plugin Activator Class
 *
 * @since 1.0.0
 */
class Bright_Tally_Activator {

    /**
     * Plugin activation method
     *
     * @since 1.0.0
     */
    public static function activate() {
        // Create database tables if needed
        self::create_tables();
        
        // Set default options
        self::set_default_options();
        
        // Set version on activation (if not already set)
        if (get_option('bright_tally_version') === false) {
            update_option('bright_tally_version', BRIGHT_TALLY_VERSION);
        }
        
        // Mark onboarding as not completed (will show on first visit)
        if (get_option('bright_tally_onboarding_completed') === false) {
            update_option('bright_tally_onboarding_completed', false);
        }
        
        // Set activation redirect flag
        set_transient('bright_tally_activation_redirect', true, 30);
        
        // Flush rewrite rules
        flush_rewrite_rules();
        
        // Log activation (only in debug mode)
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('BrightTally plugin activated successfully');
        }
    }
    
    /**
     * Create database tables
     *
     * @since 1.0.0
     */
    private static function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Create bright_tally_polls table
        $table_name = $wpdb->prefix . 'bright_tally_polls';
        $sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            poll_id varchar(255) NOT NULL,
            poll_title text NOT NULL,
            poll_data longtext NOT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY poll_id (poll_id)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        
        // Log table creation
        error_log('BrightTally database table created: ' . $table_name);
    }
    
    /**
     * Set default plugin options
     *
     * @since 1.0.0
     */
    private static function set_default_options() {
        $default_options = array(
            'bright_tally_api_key' => '',
            'bright_tally_api_url' => 'https://brighttally.com', // Base URL - API endpoints are at /api/wordpress/*
            'bright_tally_default_theme' => 'light',
            'bright_tally_cache_duration' => 300, // 5 minutes
        );
        
        foreach ($default_options as $option_name => $option_value) {
            if (get_option($option_name) === false) {
                add_option($option_name, $option_value);
            }
        }
        
        // Version is set separately above to ensure it's always updated
        
        // Log options creation (only in debug mode)
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('BrightTally default options set');
        }
    }
}
