<?php
/**
 * Plugin Name: BrightTally
 * Plugin URI: https://brighttally.com/download/bright-tally-wordpress-plugin
 * Description: Embed interactive polls, surveys and forms (coming soon!) from BrightTally directly into your WordPress site. Create engaging content with real-time results and analytics.
 * Version: 1.0.0
 * Author: Michael Loeffler (Main Street Web Developer, LLC)
 * Author URI: https://brighttally.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: bright-tally
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.4
 * Requires PHP: 7.4
 * Network: false
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('BRIGHT_TALLY_VERSION', '1.0.0');
define('BRIGHT_TALLY_PLUGIN_FILE', __FILE__);
define('BRIGHT_TALLY_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('BRIGHT_TALLY_PLUGIN_URL', plugin_dir_url(__FILE__));
define('BRIGHT_TALLY_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Include required files
require_once BRIGHT_TALLY_PLUGIN_DIR . 'includes/class-bright-tally.php';
require_once BRIGHT_TALLY_PLUGIN_DIR . 'includes/class-bright-tally-activator.php';
require_once BRIGHT_TALLY_PLUGIN_DIR . 'includes/class-bright-tally-deactivator.php';

// Register activation and deactivation hooks
register_activation_hook(__FILE__, array('Bright_Tally_Activator', 'activate'));
register_deactivation_hook(__FILE__, array('Bright_Tally_Deactivator', 'deactivate'));

/**
 * Initialize the plugin
 */
function bright_tally_init() {
    $plugin = new Bright_Tally();
    $plugin->run();
}
add_action('plugins_loaded', 'bright_tally_init');

/**
 * AJAX handlers
 */

// Test connection AJAX handler
add_action('wp_ajax_bright_tally_test_connection', 'bright_tally_test_connection');
function bright_tally_test_connection() {
    check_ajax_referer('bright_tally_test_connection', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_die('Unauthorized');
    }
    
    $api = new Bright_Tally_API();
    $result = $api->test_connection();
    
    if (is_wp_error($result)) {
        wp_send_json_error(esc_html__('Connection test failed', 'bright-tally'));
    } else {
        wp_send_json_success(esc_html__('Connection successful!', 'bright-tally'));
    }
}

// Get poll data AJAX handler
add_action('wp_ajax_bright_tally_get_poll', 'bright_tally_get_poll');
add_action('wp_ajax_nopriv_bright_tally_get_poll', 'bright_tally_get_poll');
function bright_tally_get_poll() {
    check_ajax_referer('bright_tally_public_nonce', 'nonce');
    
    if (!isset($_POST['poll_id'])) {
        wp_send_json_error(esc_html__('Poll ID is required', 'bright-tally'));
    }
    
    $poll_id = sanitize_text_field($_POST['poll_id']);
    
    if (empty($poll_id)) {
        wp_send_json_error(esc_html__('Poll ID is required', 'bright-tally'));
    }
    
    // Validate poll ID format (alphanumeric, hyphens, underscores only)
    // Also validate length to prevent DoS (max 255 chars)
    if (strlen($poll_id) > 255) {
        wp_send_json_error(esc_html__('Poll ID is too long', 'bright-tally'));
    }
    
    if (!preg_match('/^[a-zA-Z0-9_-]+$/', $poll_id)) {
        wp_send_json_error(esc_html__('Invalid poll ID format', 'bright-tally'));
    }
    
    $api = new Bright_Tally_API();
    $poll_data = $api->get_poll($poll_id);
    
    if (is_wp_error($poll_data)) {
        wp_send_json_error(esc_html__('Failed to load poll', 'bright-tally'));
    } else {
        wp_send_json_success($poll_data);
    }
}

// Submit vote AJAX handler
add_action('wp_ajax_bright_tally_submit_vote', 'bright_tally_submit_vote');
add_action('wp_ajax_nopriv_bright_tally_submit_vote', 'bright_tally_submit_vote');
function bright_tally_submit_vote() {
    check_ajax_referer('bright_tally_public_nonce', 'nonce');
    
    if (!isset($_POST['poll_id']) || !isset($_POST['option_index'])) {
        wp_send_json_error(esc_html__('Poll ID and option are required', 'bright-tally'));
    }
    
    $poll_id = sanitize_text_field($_POST['poll_id']);
    $option_index = isset($_POST['option_index']) ? intval($_POST['option_index']) : -1;
    
    // Validate poll ID
    if (empty($poll_id)) {
        wp_send_json_error(esc_html__('Poll ID is required', 'bright-tally'));
    }
    
    // Validate poll ID length
    if (strlen($poll_id) > 255) {
        wp_send_json_error(esc_html__('Poll ID is too long', 'bright-tally'));
    }
    
    // Validate poll ID format
    if (!preg_match('/^[a-zA-Z0-9_-]+$/', $poll_id)) {
        wp_send_json_error(esc_html__('Invalid poll ID format', 'bright-tally'));
    }
    
    // Validate option_index
    if ($option_index < 0) {
        wp_send_json_error(esc_html__('Invalid option index', 'bright-tally'));
    }
    
    // Validate option_index is reasonable (prevent integer overflow attacks)
    if ($option_index > 1000) {
        wp_send_json_error(esc_html__('Invalid option index', 'bright-tally'));
    }
    
    // Validate option_index is not a float or other non-integer
    if (isset($_POST['option_index']) && (string)$option_index !== (string)floatval($_POST['option_index'])) {
        wp_send_json_error(esc_html__('Invalid option index format', 'bright-tally'));
    }
    
    // Sanitize user IP and user agent
    $user_ip = isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field($_SERVER['REMOTE_ADDR']) : '';
    $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field($_SERVER['HTTP_USER_AGENT']) : '';
    
    // Validate IP format (basic validation)
    if (!empty($user_ip) && !filter_var($user_ip, FILTER_VALIDATE_IP) && !filter_var($user_ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
        // Log but don't fail - IP might be from proxy
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('BrightTally: Invalid IP format: ' . $user_ip);
        }
        $user_ip = '';
    }
    
    // Limit user agent length to prevent DoS
    if (strlen($user_agent) > 500) {
        $user_agent = substr($user_agent, 0, 500);
    }
    
    $api = new Bright_Tally_API();
    $response = $api->submit_poll_response($poll_id, array(
        'option_index' => $option_index,
        'user_ip' => $user_ip,
        'user_agent' => $user_agent,
        'timestamp' => current_time('mysql')
    ));
    
    if (is_wp_error($response)) {
        wp_send_json_error(esc_html__('Failed to submit vote', 'bright-tally'));
    } else {
        wp_send_json_success(esc_html__('Vote submitted successfully', 'bright-tally'));
    }
}

// Get poll results AJAX handler
add_action('wp_ajax_bright_tally_get_results', 'bright_tally_get_results');
add_action('wp_ajax_nopriv_bright_tally_get_results', 'bright_tally_get_results');
function bright_tally_get_results() {
    check_ajax_referer('bright_tally_public_nonce', 'nonce');
    
    if (!isset($_POST['poll_id'])) {
        wp_send_json_error(esc_html__('Poll ID is required', 'bright-tally'));
    }
    
    $poll_id = sanitize_text_field($_POST['poll_id']);
    
    if (empty($poll_id)) {
        wp_send_json_error(esc_html__('Poll ID is required', 'bright-tally'));
    }
    
    // Validate poll ID length
    if (strlen($poll_id) > 255) {
        wp_send_json_error(esc_html__('Poll ID is too long', 'bright-tally'));
    }
    
    // Validate poll ID format
    if (!preg_match('/^[a-zA-Z0-9_-]+$/', $poll_id)) {
        wp_send_json_error(esc_html__('Invalid poll ID format', 'bright-tally'));
    }
    
    $api = new Bright_Tally_API();
    $results = $api->get_poll_results($poll_id);
    
    if (is_wp_error($results)) {
        wp_send_json_error(esc_html__('Failed to load poll results', 'bright-tally'));
    } else {
        wp_send_json_success($results);
    }
}

// OAuth callback handler
add_action('wp_ajax_bright_tally_oauth_callback', 'bright_tally_oauth_callback');
function bright_tally_oauth_callback() {
    check_ajax_referer('bright_tally_oauth', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(esc_html__('Unauthorized', 'bright-tally'));
    }
    
    $code = isset($_POST['code']) ? sanitize_text_field($_POST['code']) : '';
    $state = isset($_POST['state']) ? sanitize_text_field($_POST['state']) : '';
    
    if (empty($code)) {
        wp_send_json_error(esc_html__('Authorization code is required', 'bright-tally'));
    }
    
    // Validate code length
    if (strlen($code) > 500) {
        wp_send_json_error(esc_html__('Authorization code is too long', 'bright-tally'));
    }
    
    // Validate state parameter for CSRF protection
    if (!empty($state)) {
        // Validate state length
        if (strlen($state) > 100) {
            wp_send_json_error(esc_html__('Invalid state parameter', 'bright-tally'));
        }
        
        $stored_state = get_option('bright_tally_oauth_state', '');
        if (empty($stored_state) || !hash_equals($stored_state, $state)) {
            wp_send_json_error(esc_html__('Invalid state parameter', 'bright-tally'));
        }
        // Clear state after use (one-time use)
        delete_option('bright_tally_oauth_state');
    }
    
    // Validate code format (alphanumeric and hyphens)
    if (!preg_match('/^[a-zA-Z0-9_-]+$/', $code)) {
        wp_send_json_error(esc_html__('Invalid authorization code format', 'bright-tally'));
    }
    
    // Build callback URL
    $redirect_uri = admin_url('admin-ajax.php?action=bright_tally_oauth_callback');
    
    $oauth = new Bright_Tally_OAuth();
    $result = $oauth->exchange_code_for_token($code, $redirect_uri);
    
    if (is_wp_error($result)) {
        wp_send_json_error(esc_html__('Connection failed. Please try again.', 'bright-tally'));
    }
    
    // Store token and user data
    if (isset($result['access_token'])) {
        Bright_Tally_Token_Manager::store_access_token($result['access_token']);
        
        if (isset($result['expires_in'])) {
            Bright_Tally_Token_Manager::store_token_expires_at(time() + intval($result['expires_in']));
        }
        
        if (isset($result['user'])) {
            Bright_Tally_Token_Manager::store_user_data($result['user']);
        }
        
        Bright_Tally_Token_Manager::store_connection_status('connected');
        
        wp_send_json_success(array(
            'message' => esc_html__('Account connected successfully!', 'bright-tally'),
            'user' => $result['user'] ?? null,
        ));
    } else {
        wp_send_json_error(esc_html__('Invalid response from server', 'bright-tally'));
    }
}

// Get OAuth authorization URL
add_action('wp_ajax_bright_tally_get_oauth_url', 'bright_tally_get_oauth_url');
function bright_tally_get_oauth_url() {
    if (defined('WP_DEBUG') && WP_DEBUG) {
        error_log('BrightTally: Getting OAuth URL');
    }
    check_ajax_referer('bright_tally_oauth', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(esc_html__('Unauthorized', 'bright-tally'));
    }
    
    // Generate state for CSRF protection
    $state = wp_generate_password(32, false);
    if (empty($state) || strlen($state) < 16) {
        wp_send_json_error(esc_html__('Failed to generate security token', 'bright-tally'));
    }
    update_option('bright_tally_oauth_state', $state, false);
    
    // Build callback URL - use admin page for proper redirect handling
    $redirect_uri = admin_url('admin.php?page=bright-tally-settings&oauth_callback=1');
    
    // Validate redirect URI is not empty
    if (empty($redirect_uri)) {
        wp_send_json_error(esc_html__('Invalid redirect URI', 'bright-tally'));
    }

    if (defined('WP_DEBUG') && WP_DEBUG) {
        error_log('BrightTally: Redirect URI=' . $redirect_uri);
    }
    
    $oauth = new Bright_Tally_OAuth();
    $auth_url = $oauth->get_authorization_url($redirect_uri, $state);

    if (is_wp_error($auth_url)) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('BrightTally: Auth URL error=' . $auth_url->get_error_message());
        }
        // Check if login is required
        if ($auth_url->get_error_code() === 'requires_login') {
            $error_data = $auth_url->get_error_data();
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('BrightTally: Login URL=' . $error_data['login_url']);
            }
            wp_send_json_error(array(
                'requires_login' => true,
                'login_url' => esc_url($error_data['login_url'] ?? ''),
                'message' => esc_html($auth_url->get_error_message()),
            ));
        }
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('BrightTally: Auth URL error=' . $auth_url->get_error_message());
        }
        wp_send_json_error(array(
            'message' => esc_html($auth_url->get_error_message()),
            'error_code' => esc_html($auth_url->get_error_code()),
        ));
    }
    
    if (empty($auth_url)) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('BrightTally: No authorization URL received from BrightTally API. Please check your API URL configuration.');
        }
        wp_send_json_error(array(
            'message' => esc_html__('No authorization URL received from BrightTally API. Please check your API URL configuration.', 'bright-tally'),
        ));
    }
    
    if (defined('WP_DEBUG') && WP_DEBUG) {
        error_log('BrightTally: Sending JSON success response');
        error_log('BrightTally: Auth URL=' . $auth_url);
        error_log('BrightTally: State=' . $state);
    }
    wp_send_json_success(array(
        'auth_url' => $auth_url,
        'state' => $state,
    ));
}

// Login with email/password
add_action('wp_ajax_bright_tally_login', 'bright_tally_login');
function bright_tally_login() {
    check_ajax_referer('bright_tally_create_account', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(esc_html__('Unauthorized', 'bright-tally'));
    }
    
    if (!isset($_POST['email']) || !isset($_POST['password'])) {
        wp_send_json_error(esc_html__('Email and password are required', 'bright-tally'));
    }
    
    $email = sanitize_email($_POST['email']);
    $password = isset($_POST['password']) ? $_POST['password'] : '';
    
    // Validate email format
    if (!is_email($email)) {
        wp_send_json_error(esc_html__('Invalid email address', 'bright-tally'));
    }
    
    // Validate email length
    if (strlen($email) > 255) {
        wp_send_json_error(esc_html__('Email address is too long', 'bright-tally'));
    }
    
    // Validate password is not empty and reasonable length
    if (empty($password)) {
        wp_send_json_error(esc_html__('Password is required', 'bright-tally'));
    }
    
    // Limit password length to prevent DoS
    if (strlen($password) > 1000) {
        wp_send_json_error(esc_html__('Password is too long', 'bright-tally'));
    }
    
    // Validate password contains at least some characters (not just whitespace)
    if (strlen(trim($password)) === 0) {
        wp_send_json_error(esc_html__('Password cannot be empty', 'bright-tally'));
    }
    
    $oauth = new Bright_Tally_OAuth();
    $result = $oauth->login($email, $password);
    
    if (is_wp_error($result)) {
        wp_send_json_error(esc_html__('Login failed. Please check your credentials.', 'bright-tally'));
    }
    
    // Store token and user data
    if (isset($result['access_token'])) {
        Bright_Tally_Token_Manager::store_access_token($result['access_token']);
        
        if (isset($result['expires_in'])) {
            Bright_Tally_Token_Manager::store_token_expires_at(time() + intval($result['expires_in']));
        }
        
        if (isset($result['user'])) {
            Bright_Tally_Token_Manager::store_user_data($result['user']);
        }
        
        Bright_Tally_Token_Manager::store_connection_status('connected');
        
        wp_send_json_success(array(
            'message' => esc_html__('Account connected successfully!', 'bright-tally'),
            'user' => $result['user'] ?? null,
        ));
    } else {
        wp_send_json_error(esc_html__('Invalid response from server', 'bright-tally'));
    }
}

// Create account from WordPress
add_action('wp_ajax_bright_tally_create_account', 'bright_tally_create_account');
function bright_tally_create_account() {
    check_ajax_referer('bright_tally_create_account', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(esc_html__('Unauthorized', 'bright-tally'));
    }
    
    if (!isset($_POST['name']) || !isset($_POST['email']) || !isset($_POST['password'])) {
        wp_send_json_error(esc_html__('Name, email, and password are required', 'bright-tally'));
    }
    
    $name = sanitize_text_field($_POST['name']);
    $email = sanitize_email($_POST['email']);
    $password = isset($_POST['password']) ? $_POST['password'] : '';
    
    // Validate inputs
    if (empty($name) || empty($email) || empty($password)) {
        wp_send_json_error(esc_html__('Name, email, and password are required', 'bright-tally'));
    }
    
    // Validate name - check for empty after trim
    $name = trim($name);
    if (empty($name)) {
        wp_send_json_error(esc_html__('Name cannot be empty', 'bright-tally'));
    }
    
    // Validate name length
    if (strlen($name) > 255) {
        wp_send_json_error(esc_html__('Name is too long', 'bright-tally'));
    }
    
    // Validate name doesn't contain only special characters
    if (preg_match('/^[^a-zA-Z0-9]+$/', $name)) {
        wp_send_json_error(esc_html__('Name must contain at least one letter or number', 'bright-tally'));
    }
    
    // Validate email format
    if (!is_email($email)) {
        wp_send_json_error(esc_html__('Invalid email address', 'bright-tally'));
    }
    
    // Validate email length
    if (strlen($email) > 255) {
        wp_send_json_error(esc_html__('Email address is too long', 'bright-tally'));
    }
    
    // Validate password
    if (strlen($password) < 8) {
        wp_send_json_error(esc_html__('Password must be at least 8 characters long', 'bright-tally'));
    }
    
    // Limit password length to prevent DoS
    if (strlen($password) > 1000) {
        wp_send_json_error(esc_html__('Password is too long', 'bright-tally'));
    }
    
    // Validate password contains non-whitespace characters
    if (strlen(trim($password)) < 8) {
        wp_send_json_error(esc_html__('Password must contain at least 8 non-whitespace characters', 'bright-tally'));
    }
    
    $oauth = new Bright_Tally_OAuth();
    $result = $oauth->create_account(array(
        'name' => $name,
        'email' => $email,
        'password' => $password,
    ));
    
    if (is_wp_error($result)) {
        wp_send_json_error(esc_html__('Account creation failed. Please try again.', 'bright-tally'));
    }
    
    // Store token and user data
    if (isset($result['access_token'])) {
        Bright_Tally_Token_Manager::store_access_token($result['access_token']);
        
        if (isset($result['expires_in'])) {
            Bright_Tally_Token_Manager::store_token_expires_at(time() + intval($result['expires_in']));
        }
        
        if (isset($result['user'])) {
            Bright_Tally_Token_Manager::store_user_data($result['user']);
        }
        
        Bright_Tally_Token_Manager::store_connection_status('connected');
        
        wp_send_json_success(array(
            'message' => esc_html__('Account created and connected successfully!', 'bright-tally'),
            'user' => $result['user'] ?? null,
        ));
    } else {
        wp_send_json_error(esc_html__('Invalid response from server', 'bright-tally'));
    }
}

// Disconnect account
add_action('wp_ajax_bright_tally_disconnect', 'bright_tally_disconnect');
function bright_tally_disconnect() {
    check_ajax_referer('bright_tally_oauth', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(esc_html__('Unauthorized', 'bright-tally'));
    }
    
    $oauth = new Bright_Tally_OAuth();
    $result = $oauth->revoke_token();
    
    if (is_wp_error($result)) {
        // Still clear local data even if remote revocation fails
        Bright_Tally_Token_Manager::clear_all();
        wp_send_json_success(array('message' => esc_html__('Account disconnected locally', 'bright-tally')));
    }
    
    Bright_Tally_Token_Manager::clear_all();
    wp_send_json_success(array('message' => esc_html__('Account disconnected successfully', 'bright-tally')));
}

// Get connection status
add_action('wp_ajax_bright_tally_get_connection_status', 'bright_tally_get_connection_status');
function bright_tally_get_connection_status() {
    check_ajax_referer('bright_tally_oauth', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
    }
    
    $is_connected = Bright_Tally_Token_Manager::is_connected();
    $user_data = Bright_Tally_Token_Manager::get_user_data();
    $status = Bright_Tally_Token_Manager::get_connection_status();
    
    wp_send_json_success(array(
        'connected' => $is_connected,
        'status' => $status,
        'user' => $user_data,
        'api_url' => Bright_Tally_URL_Helper::get_api_url(),
    ));
}

// Get polls via AJAX
add_action('wp_ajax_bright_tally_get_polls', 'bright_tally_get_polls_ajax');
function bright_tally_get_polls_ajax() {
    check_ajax_referer('bright_tally_oauth', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(esc_html__('Unauthorized', 'bright-tally'));
    }
    
    $api = new Bright_Tally_API();
    
    if (!$api->is_configured()) {
        wp_send_json_error(esc_html__('API not configured', 'bright-tally'));
    }
    
    $polls = $api->get_polls();
    
    if (is_wp_error($polls)) {
        wp_send_json_error(esc_html__('Failed to load polls', 'bright-tally'));
    }
    
    wp_send_json_success($polls);
}

// Complete onboarding
add_action('wp_ajax_bright_tally_complete_onboarding', 'bright_tally_complete_onboarding');
function bright_tally_complete_onboarding() {
    check_ajax_referer('bright_tally_oauth', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(esc_html__('Unauthorized', 'bright-tally'));
    }
    
    update_option('bright_tally_onboarding_completed', true);
    wp_send_json_success(array('message' => esc_html__('Onboarding completed', 'bright-tally')));
}

// Skip onboarding
add_action('wp_ajax_bright_tally_skip_onboarding', 'bright_tally_skip_onboarding');
function bright_tally_skip_onboarding() {
    check_ajax_referer('bright_tally_oauth', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(esc_html__('Unauthorized', 'bright-tally'));
    }
    
    update_option('bright_tally_onboarding_completed', true);
    wp_send_json_success(array('message' => esc_html__('Onboarding skipped', 'bright-tally')));
}

