/**
 * Edit component for BrightTally Poll Block
 *
 * @package Bright_Tally
 * @since 1.0.0
 */

import { __ } from '@wordpress/i18n';
import { useBlockProps, InspectorControls } from '@wordpress/block-editor';
import { PanelBody, SelectControl, ToggleControl, TextControl, Button, Placeholder, Spinner } from '@wordpress/components';
import { useState, useEffect } from '@wordpress/element';
import apiFetch from '@wordpress/api-fetch';

export default function Edit({ attributes, setAttributes }) {
	const { pollId, pollSlug, theme, showResults, allowVoting, width, height } = attributes;
	const [polls, setPolls] = useState([]);
	const [loading, setLoading] = useState(false);
	const [error, setError] = useState(null);
	const [showPollPicker, setShowPollPicker] = useState(!pollId);

	const blockProps = useBlockProps({
		className: 'bright-tally-block',
	});

	// Fetch polls on mount
	useEffect(() => {
		if (showPollPicker && polls.length === 0) {
			fetchPolls();
		}
	}, []);

	const fetchPolls = async () => {
		setLoading(true);
		setError(null);

		try {
			const response = await apiFetch({
				path: '/bright-tally/v1/polls',
			});

			if (response && response.success && response.data) {
				setPolls(response.data.data || []);
			} else {
				setError(response && response.data ? response.data : __('Failed to load polls', 'bright-tally'));
			}
		} catch (err) {
			setError(err.message || __('Error loading polls. Please check your connection.', 'bright-tally'));
		} finally {
			setLoading(false);
		}
	};

	const selectPoll = (poll) => {
		setAttributes({
			pollId: poll.id.toString(),
			pollSlug: poll.slug || poll.id.toString(),
		});
		setShowPollPicker(false);
	};

	const renderPollPicker = () => {
		if (loading) {
			return (
				<Placeholder icon="chart-bar" label={__('BrightTally Poll', 'bright-tally')}>
					<Spinner />
					<p>{__('Loading polls...', 'bright-tally')}</p>
				</Placeholder>
			);
		}

		if (error) {
			return (
				<Placeholder icon="chart-bar" label={__('BrightTally Poll', 'bright-tally')}>
					<p style={{ color: '#d63638' }}>{error}</p>
					<Button variant="primary" onClick={fetchPolls}>
						{__('Retry', 'bright-tally')}
					</Button>
					<p style={{ marginTop: '16px' }}>
						{__('Make sure your BrightTally account is connected in the plugin settings.', 'bright-tally')}
					</p>
				</Placeholder>
			);
		}

		if (polls.length === 0) {
			return (
				<Placeholder icon="chart-bar" label={__('BrightTally Poll', 'bright-tally')}>
					<p>{__('No polls found. Create a poll in your BrightTally dashboard.', 'bright-tally')}</p>
					<Button variant="primary" href="https://brighttally.com/dashboard/polls/create" target="_blank">
						{__('Create Poll', 'bright-tally')}
					</Button>
				</Placeholder>
			);
		}

		return (
			<Placeholder icon="chart-bar" label={__('BrightTally Poll', 'bright-tally')}>
				<div className="bright-tally-poll-picker">
					<div className="poll-picker-header" style={{ marginBottom: '16px', display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
						<h3 style={{ margin: 0 }}>{__('Select a Poll', 'bright-tally')}</h3>
						<Button variant="secondary" isSmall onClick={fetchPolls}>
							{__('Refresh', 'bright-tally')}
						</Button>
					</div>
					<div className="poll-picker-grid" style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fill, minmax(200px, 1fr))', gap: '12px', maxHeight: '400px', overflowY: 'auto' }}>
						{polls.map((poll) => (
							<div
								key={poll.id}
								className="poll-picker-card"
								onClick={() => selectPoll(poll)}
								style={{
									border: '1px solid #ddd',
									borderRadius: '4px',
									padding: '12px',
									cursor: 'pointer',
									transition: 'all 0.2s',
									background: '#fff'
								}}
								onMouseEnter={(e) => e.currentTarget.style.borderColor = '#0073aa'}
								onMouseLeave={(e) => e.currentTarget.style.borderColor = '#ddd'}
							>
								<div className="poll-card-header" style={{ marginBottom: '8px' }}>
									<span style={{
										display: 'inline-block',
										padding: '2px 6px',
										borderRadius: '3px',
										fontSize: '11px',
										fontWeight: 600,
										textTransform: 'uppercase',
										background: poll.status === 'active' ? '#d4edda' : '#fff3cd',
										color: poll.status === 'active' ? '#155724' : '#856404'
									}}>
										{poll.status || 'Active'}
									</span>
								</div>
								<div className="poll-card-body">
									<h4 style={{ margin: '0 0 4px 0', fontSize: '14px', fontWeight: 600 }}>{poll.title}</h4>
									{poll.description && (
										<p style={{ margin: '0 0 8px 0', color: '#666', fontSize: '12px' }}>
											{poll.description.substring(0, 80)}...
										</p>
									)}
									<div className="poll-stats" style={{ fontSize: '11px', color: '#999' }}>
										<span>{poll.response_count || 0} {__('responses', 'bright-tally')}</span>
									</div>
								</div>
							</div>
						))}
					</div>
				</div>
			</Placeholder>
		);
	};

	const renderPollPreview = () => {
		if (!pollId) {
			return renderPollPicker();
		}

		const apiUrl = window.brightTallyApiUrl || 'https://brighttally.com';
		const embedUrl = `${apiUrl}/embed/${pollSlug || pollId}`;
		const iframeHeight = height === 'auto' ? '600' : height;

		return (
			<div {...blockProps}>
				<div className="bright-tally-preview" style={{ padding: '16px', border: '1px solid #ddd', borderRadius: '4px', background: '#fff' }}>
					<div className="preview-header" style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginBottom: '12px' }}>
						<h3 style={{ margin: 0, fontSize: '16px' }}>{__('Poll Preview', 'bright-tally')}</h3>
						<Button variant="link" onClick={() => setShowPollPicker(true)}>
							{__('Change Poll', 'bright-tally')}
						</Button>
					</div>
					<div className="preview-content" style={{ marginBottom: '12px', border: '1px solid #ddd', borderRadius: '4px', overflow: 'hidden' }}>
						<iframe
							src={embedUrl}
							width={width}
							height={iframeHeight}
							frameBorder="0"
							title={__('Poll Preview', 'bright-tally')}
							style={{ display: 'block', width: '100%', border: 'none' }}
						/>
					</div>
					<div className="preview-shortcode" style={{ padding: '8px', background: '#f5f5f5', borderRadius: '4px' }}>
						<code style={{ fontSize: '11px', color: '#333' }}>
							{`[bright_tally id="${pollSlug || pollId}" theme="${theme}" show_results="${showResults ? 'true' : 'false'}" allow_voting="${allowVoting ? 'true' : 'false'}"]`}
						</code>
					</div>
				</div>
			</div>
		);
	};

	return (
		<>
			<InspectorControls>
				<PanelBody title={__('Poll Settings', 'bright-tally')} initialOpen={true}>
					{!pollId && (
						<p className="description" style={{ fontStyle: 'italic', color: '#666' }}>
							{__('Select a poll to configure settings.', 'bright-tally')}
						</p>
					)}
					{pollId && (
						<>
							<Button
								variant="secondary"
								onClick={() => setShowPollPicker(true)}
								style={{ width: '100%', marginBottom: '16px' }}
							>
								{__('Select Different Poll', 'bright-tally')}
							</Button>
							<SelectControl
								label={__('Theme', 'bright-tally')}
								value={theme}
								options={[
									{ label: __('Light', 'bright-tally'), value: 'light' },
									{ label: __('Dark', 'bright-tally'), value: 'dark' },
								]}
								onChange={(value) => setAttributes({ theme: value })}
							/>
							<ToggleControl
								label={__('Show Results', 'bright-tally')}
								checked={showResults}
								onChange={(value) => setAttributes({ showResults: value })}
							/>
							<ToggleControl
								label={__('Allow Voting', 'bright-tally')}
								checked={allowVoting}
								onChange={(value) => setAttributes({ allowVoting: value })}
							/>
							<TextControl
								label={__('Width', 'bright-tally')}
								value={width}
								onChange={(value) => setAttributes({ width: value })}
								help={__('e.g., 100%, 600px', 'bright-tally')}
							/>
							<TextControl
								label={__('Height', 'bright-tally')}
								value={height}
								onChange={(value) => setAttributes({ height: value })}
								help={__('e.g., auto, 600px', 'bright-tally')}
							/>
						</>
					)}
				</PanelBody>
			</InspectorControls>

			{showPollPicker ? renderPollPicker() : renderPollPreview()}
		</>
	);
}

