/**
 * Public JavaScript for BrightTally plugin
 *
 * @package Bright_Tally
 * @since 1.0.0
 */

(function($) {
    'use strict';
    
    // Global BrightTally object
    window.BrightTally = {
        polls: {},
        
        init: function() {
            this.bindEvents();
        },
        
        initPoll: function(instanceId) {
            var $poll = $('#' + instanceId);
            if ($poll.length === 0) return;
            
            var pollId = $poll.data('poll-id');
            var allowVoting = $poll.data('allow-voting') === true;
            var showResults = $poll.data('show-results') === true;
            
            // Store poll instance
            this.polls[instanceId] = {
                element: $poll,
                pollId: pollId,
                allowVoting: allowVoting,
                showResults: showResults,
                hasVoted: false
            };
            
            // Load poll data
            this.loadPollData(instanceId);
            
            // Bind form submission
            if (allowVoting) {
                this.bindPollForm(instanceId);
            }
        },
        
        loadPollData: function(instanceId) {
            var poll = this.polls[instanceId];
            if (!poll) return;
            
            var $poll = poll.element;
            var $loading = $poll.find('.bright-tally-loading');
            var $content = $poll.find('.bright-tally-content');
            var $error = $poll.find('.bright-tally-error');
            
            // Show loading state
            $loading.show();
            $content.hide();
            $error.hide();
            
            // Make AJAX request to get poll data
            $.ajax({
                url: bright_tally_public_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'bright_tally_get_poll',
                    poll_id: poll.pollId,
                    nonce: bright_tally_public_ajax.nonce
                },
                success: function(response) {
                    if (response.success) {
                        poll.data = response.data;
                        BrightTally.renderPoll(instanceId);
                    } else {
                        BrightTally.showError(instanceId, response.data || 'Failed to load poll');
                    }
                },
                error: function(xhr, status, error) {
                    BrightTally.showError(instanceId, 'Network error: ' + error);
                }
            });
        },
        
        renderPoll: function(instanceId) {
            var poll = this.polls[instanceId];
            if (!poll || !poll.data) return;
            
            var $poll = poll.element;
            var $loading = $poll.find('.bright-tally-loading');
            var $content = $poll.find('.bright-tally-content');
            var $title = $poll.find('.bright-tally-title');
            var $description = $poll.find('.bright-tally-description');
            var $options = $poll.find('.bright-tally-options');
            
            // Update title and description
            $title.text(poll.data.title);
            if (poll.data.description) {
                $description.text(poll.data.description).show();
            } else {
                $description.hide();
            }
            
            // Render options
            this.renderOptions(instanceId);
            
            // Show content
            $loading.hide();
            $content.show();
        },
        
        renderOptions: function(instanceId) {
            var poll = this.polls[instanceId];
            if (!poll || !poll.data || !poll.data.options) return;
            
            var $form = poll.element.find('.bright-tally-form');
            var $optionsContainer = $form.find('.bright-tally-options');
            
            // Clear existing options
            $optionsContainer.find('.bright-tally-option').remove();
            
            // Add new options
            poll.data.options.forEach(function(option, index) {
                var $option = $('<div class="bright-tally-option"></div>');
                var $label = $('<label></label>');
                var $input = $('<input type="radio" name="poll_option" value="' + index + '" class="bright-tally-radio">');
                var $text = $('<span class="bright-tally-option-text">' + option.text + '</span>');
                
                $label.append($input).append($text);
                $option.append($label);
                $optionsContainer.append($option);
            });
        },
        
        bindPollForm: function(instanceId) {
            var poll = this.polls[instanceId];
            if (!poll) return;
            
            var $form = poll.element.find('.bright-tally-form');
            
            $form.off('submit.brightTally').on('submit.brightTally', function(e) {
                e.preventDefault();
                BrightTally.submitVote(instanceId);
            });
        },
        
        submitVote: function(instanceId) {
            var poll = this.polls[instanceId];
            if (!poll || poll.hasVoted) return;
            
            var $form = poll.element.find('.bright-tally-form');
            var selectedOption = $form.find('input[name="poll_option"]:checked').val();
            
            if (!selectedOption) {
                this.showError(instanceId, 'Please select an option');
                return;
            }
            
            var $submitButton = $form.find('.bright-tally-submit');
            var originalText = $submitButton.text();
            
            // Disable form and show loading
            $submitButton.prop('disabled', true).text('Submitting...');
            
            // Make AJAX request
            $.ajax({
                url: bright_tally_public_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'bright_tally_submit_vote',
                    poll_id: poll.pollId,
                    option_index: selectedOption,
                    nonce: bright_tally_public_ajax.nonce
                },
                success: function(response) {
                    if (response.success) {
                        poll.hasVoted = true;
                        BrightTally.showSuccess(instanceId, 'Thank you for voting!');
                        
                        // Hide form and show results if enabled
                        if (poll.showResults) {
                            BrightTally.loadResults(instanceId);
                        } else {
                            $form.hide();
                        }
                    } else {
                        BrightTally.showError(instanceId, response.data || 'Failed to submit vote');
                    }
                },
                error: function(xhr, status, error) {
                    BrightTally.showError(instanceId, 'Network error: ' + error);
                },
                complete: function() {
                    $submitButton.prop('disabled', false).text(originalText);
                }
            });
        },
        
        loadResults: function(instanceId) {
            var poll = this.polls[instanceId];
            if (!poll) return;
            
            var $results = poll.element.find('.bright-tally-results');
            var $resultsContent = $results.find('.bright-tally-results-content');
            
            // Show loading in results
            $resultsContent.html('<p>Loading results...</p>');
            $results.show();
            
            // Make AJAX request for results
            $.ajax({
                url: bright_tally_public_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'bright_tally_get_results',
                    poll_id: poll.pollId,
                    nonce: bright_tally_public_ajax.nonce
                },
                success: function(response) {
                    if (response.success) {
                        BrightTally.renderResults(instanceId, response.data);
                    } else {
                        $resultsContent.html('<p>Failed to load results</p>');
                    }
                },
                error: function() {
                    $resultsContent.html('<p>Failed to load results</p>');
                }
            });
        },
        
        renderResults: function(instanceId, results) {
            var poll = this.polls[instanceId];
            if (!poll || !results) return;
            
            var $resultsContent = poll.element.find('.bright-tally-results-content');
            var html = '';
            
            if (results.options && results.options.length > 0) {
                var totalVotes = results.total_votes || 0;
                
                results.options.forEach(function(option, index) {
                    var votes = option.votes || 0;
                    var percentage = totalVotes > 0 ? Math.round((votes / totalVotes) * 100) : 0;
                    
                    html += '<div class="bright-tally-result-item">';
                    html += '<div class="bright-tally-result-label">' + option.text + '</div>';
                    html += '<div class="bright-tally-result-bar">';
                    html += '<div class="bright-tally-result-fill" style="width: ' + percentage + '%"></div>';
                    html += '</div>';
                    html += '<div class="bright-tally-result-percentage">' + percentage + '%</div>';
                    html += '</div>';
                });
            } else {
                html = '<p>No results available</p>';
            }
            
            $resultsContent.html(html);
        },
        
        showError: function(instanceId, message) {
            var poll = this.polls[instanceId];
            if (!poll) return;
            
            var $error = poll.element.find('.bright-tally-error');
            $error.html('<p>' + message + '</p>').show();
            
            // Hide other sections
            poll.element.find('.bright-tally-loading, .bright-tally-content').hide();
        },
        
        showSuccess: function(instanceId, message) {
            var poll = this.polls[instanceId];
            if (!poll) return;
            
            var $content = poll.element.find('.bright-tally-content');
            var $success = $('<div class="bright-tally-success"><p>' + message + '</p></div>');
            
            $content.prepend($success);
            
            // Auto-hide success message after 3 seconds
            setTimeout(function() {
                $success.fadeOut();
            }, 3000);
        },
        
        bindEvents: function() {
            // Initialize all polls on the page
            $('.bright-tally-poll').each(function() {
                var instanceId = $(this).attr('id');
                if (instanceId) {
                    BrightTally.initPoll(instanceId);
                }
            });
        }
    };
    
    // Initialize when document is ready
    $(document).ready(function() {
        BrightTally.init();
    });
    
    // Re-initialize on AJAX content load (for dynamic content)
    $(document).on('DOMNodeInserted', function(e) {
        if ($(e.target).find('.bright-tally-poll').length > 0) {
            $(e.target).find('.bright-tally-poll').each(function() {
                var instanceId = $(this).attr('id');
                if (instanceId && !BrightTally.polls[instanceId]) {
                    BrightTally.initPoll(instanceId);
                }
            });
        }
    });
    
})(jQuery);
