/**
 * Admin JavaScript for BrightTally plugin
 *
 * @package Bright_Tally
 * @since 1.0.0
 */

(function($) {
    'use strict';
    
    // Initialize when document is ready
    $(document).ready(function() {
        initAdmin();
    });
    
    function initAdmin() {
        // Initialize tooltips
        initTooltips();
        
        // Initialize modals
        initModals();
        
        // Initialize AJAX handlers
        initAjaxHandlers();
    }
    
    function initTooltips() {
        // Add tooltip functionality if needed
        $('[data-tooltip]').each(function() {
            var tooltip = $(this).data('tooltip');
            $(this).attr('title', tooltip);
        });
    }
    
    function initModals() {
        // Close modal when clicking outside
        $(document).on('click', '.bright-tally-modal', function(e) {
            if (e.target === this) {
                $(this).hide();
            }
        });
        
        // Close modal when clicking close button
        $(document).on('click', '.bright-tally-modal-close', function() {
            $(this).closest('.bright-tally-modal').hide();
        });
        
        // Close modal with Escape key
        $(document).on('keydown', function(e) {
            if (e.keyCode === 27) { // Escape key
                $('.bright-tally-modal:visible').hide();
            }
        });
    }
    
    function initAjaxHandlers() {
        // Test connection handler
        $(document).on('click', '#test-connection', function(e) {
            e.preventDefault();
            testConnection();
        });
        
        // Copy shortcode handler
        $(document).on('click', '.copy-shortcode', function(e) {
            e.preventDefault();
            copyToClipboard($(this).siblings('input'));
        });
        
        // Get shortcode handler
        $(document).on('click', '.get-shortcode', function(e) {
            e.preventDefault();
            var pollId = $(this).data('poll-id');
            var pollSlug = $(this).data('poll-slug') || pollId;
            showShortcodeModal(pollId, pollSlug);
        });
        
        // Update shortcode based on options
        $(document).on('change', '.bright-tally-shortcode-options input', function() {
            updateShortcode();
        });
        
        // Connection interface handlers
        $(document).on('click', '#quick-connect', function(e) {
            e.preventDefault();
            initiateQuickConnect();
        });
        
        $(document).on('click', '#manual-setup', function(e) {
            e.preventDefault();
            showManualConfig();
        });
        
        // Account creation handlers
        $(document).on('click', '#email-login-btn', function(e) {
            e.preventDefault();
            showEmailLoginForm();
        });
        
        $(document).on('click', '#create-account-btn', function(e) {
            e.preventDefault();
            showCreateAccountForm();
        });
        
        $(document).on('submit', '#email-login-account-form', function(e) {
            e.preventDefault();
            loginWithEmail();
        });
        
        $(document).on('submit', '#create-account-form form', function(e) {
            e.preventDefault();
            createAccount();
        });
        
        // Disconnect handler
        $(document).on('click', '.bright-tally-disconnect', function(e) {
            e.preventDefault();
            disconnectAccount();
        });
        
        // Help panel toggle
        $(document).on('click', '[data-toggle="docs-help"]', function(e) {
            e.preventDefault();
            var $panel = $('#bright-tally-help-panel');
            if ($panel.length) {
                $panel.slideToggle(200);
            }
        });
        
        // Help search
        $(document).on('submit', '#bright-tally-help-search', function(e) {
            e.preventDefault();
            var query = $('#bright-tally-help-query').val().trim();
            var docsBase = getDocsBaseUrl();
            var targetUrl = docsBase;
            
            if (query.length) {
                var separator = docsBase.indexOf('?') === -1 ? '?' : '&';
                targetUrl += separator + 'search=' + encodeURIComponent(query);
            }
            
            window.open(targetUrl, '_blank', 'noopener,noreferrer');
        });
        
        // Check connection status on page load
        checkConnectionStatus();
        
        // Onboarding handlers
        initOnboarding();
        
        // Poll picker enhancements
        initPollPicker();
    }
    
    function initPollPicker() {
        // Refresh polls button
        $(document).on('click', '#refresh-polls', function(e) {
            e.preventDefault();
            refreshPolls();
        });
        
        // Auto-refresh polls every 30 seconds if on polls page
        if ($('#polls-grid').length) {
            setInterval(function() {
                // Only refresh if user is not actively interacting
                if (!$('#poll-search').is(':focus') && !$('.bright-tally-modal:visible').length) {
                    refreshPolls(true); // Silent refresh
                }
            }, 30000);
        }
    }
    
    function refreshPolls(silent) {
        silent = silent || false;
        var $grid = $('#polls-grid');
        var $loading = $('#polls-loading');
        
        if (!$grid.length) return;
        
        if (!silent) {
            $loading.show();
            $grid.fadeOut(200);
        }
        
        // Fetch polls via AJAX
        $.ajax({
            url: bright_tally_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'bright_tally_get_polls',
                nonce: bright_tally_ajax.oauth_nonce
            },
            success: function(response) {
                if (response.success && response.data && response.data.data) {
                    // Update poll cards
                    updatePollCards(response.data.data);
                    if (!silent) {
                        $loading.hide();
                        $grid.fadeIn(200);
                        showNotice('Polls refreshed', 'success');
                    }
                } else {
                    if (!silent) {
                        $loading.hide();
                        $grid.fadeIn(200);
                        showNotice('Failed to refresh polls', 'error');
                    }
                }
            },
            error: function() {
                if (!silent) {
                    $loading.hide();
                    $grid.fadeIn(200);
                    showNotice('Error refreshing polls', 'error');
                }
            }
        });
    }
    
    function updatePollCards(polls) {
        // Helper function to format numbers
        function number_format(num) {
            return num.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ",");
        }
        
        // Update poll stats without full page reload
        polls.forEach(function(poll) {
            var $card = $('.bright-tally-poll-card[data-poll-id="' + poll.id + '"]');
            if ($card.length) {
                // Update response count
                var $statValue = $card.find('.stat-value').first();
                if ($statValue.length) {
                    $statValue.text(number_format(poll.response_count || 0));
                }
                
                // Update status if changed
                var currentStatus = $card.data('status');
                var newStatus = (poll.status || (poll.is_active ? 'active' : 'inactive')).toLowerCase();
                if (currentStatus !== newStatus) {
                    $card.attr('data-status', newStatus);
                    $card.find('.bright-tally-poll-status-badge')
                        .removeClass('status-' + currentStatus)
                        .addClass('status-' + newStatus);
                    $card.find('.status-text').text(newStatus.charAt(0).toUpperCase() + newStatus.slice(1));
                }
            }
        });
    }
    
    function initOnboarding() {
        // Onboarding quick connect
        $(document).on('click', '#onboarding-quick-connect', function(e) {
            e.preventDefault();
            var $button = $(this);
            var originalText = $button.text();
            $button.prop('disabled', true).text('Connecting...');
            
            initiateQuickConnect();
            
            // Monitor connection status
            var checkInterval = setInterval(function() {
                $.ajax({
                    url: bright_tally_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'bright_tally_get_connection_status',
                        nonce: bright_tally_ajax.oauth_nonce
                    },
                    success: function(response) {
                        if (response.success && response.data.connected) {
                            clearInterval(checkInterval);
                            $button.prop('disabled', false).text(originalText);
                            // Move to next step
                            setTimeout(function() {
                                brightTallyOnboarding.nextStep(2);
                            }, 1000);
                        }
                    }
                });
            }, 2000);
            
            // Stop checking after 2 minutes
            setTimeout(function() {
                clearInterval(checkInterval);
                $button.prop('disabled', false).text(originalText);
            }, 120000);
        });
        
        // Onboarding create account
        $(document).on('click', '#onboarding-create-account', function(e) {
            e.preventDefault();
            $('#onboarding-create-account-form').slideDown(300, function() {
                $('#onboarding_name').focus();
            });
        });
        
        // Onboarding account form submission
        $(document).on('submit', '#onboarding-account-form', function(e) {
            e.preventDefault();
            createAccountFromOnboarding();
        });
        
        // Auto-check connection status on step 1
        if (brightTallyOnboarding.currentStep() === 1) {
            checkConnectionStatus();
        }
    }
    
    function createAccountFromOnboarding() {
        var form = $('#onboarding-account-form');
        var name = form.find('#onboarding_name').val().trim();
        var email = form.find('#onboarding_email').val().trim();
        var password = form.find('#onboarding_password').val();
        var passwordConfirm = form.find('#onboarding_password_confirm').val();
        
        // Clear previous errors
        form.find('.error-message').remove();
        form.find('.bright-tally-input').removeClass('error');
        
        // Validation
        var hasError = false;
        
        if (!name) {
            showFieldError('#onboarding_name', 'Name is required');
            hasError = true;
        }
        
        if (!email) {
            showFieldError('#onboarding_email', 'Email is required');
            hasError = true;
        } else if (!isValidEmail(email)) {
            showFieldError('#onboarding_email', 'Please enter a valid email address');
            hasError = true;
        }
        
        if (!password) {
            showFieldError('#onboarding_password', 'Password is required');
            hasError = true;
        } else if (password.length < 8) {
            showFieldError('#onboarding_password', 'Password must be at least 8 characters long');
            hasError = true;
        }
        
        if (!passwordConfirm) {
            showFieldError('#onboarding_password_confirm', 'Please confirm your password');
            hasError = true;
        } else if (password !== passwordConfirm) {
            showFieldError('#onboarding_password_confirm', 'Passwords do not match');
            hasError = true;
        }
        
        if (hasError) {
            return;
        }
        
        var submitButton = form.find('button[type="submit"]');
        var originalText = submitButton.text();
        submitButton.prop('disabled', true).text('Creating Account...');
        
        // Show loading state
        var $stepContent = form.closest('.onboarding-step-content');
        $stepContent.addClass('loading');
        
        $.ajax({
            url: bright_tally_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'bright_tally_create_account',
                name: name,
                email: email,
                password: password,
                nonce: bright_tally_ajax.create_account_nonce
            },
            success: function(response) {
                if (response.success) {
                    // Show success message
                    $stepContent.removeClass('loading').addClass('success');
                    
                    // Update step indicator
                    var $step1 = $('.bright-tally-step-indicator .step').first();
                    $step1.addClass('completed');
                    
                    // Move to next step after short delay
                    setTimeout(function() {
                        brightTallyOnboarding.nextStep(2);
                    }, 1500);
                } else {
                    $stepContent.removeClass('loading');
                    var errorMsg = response.data || 'Account creation failed';
                    if (typeof errorMsg === 'object' && errorMsg.errors) {
                        // Handle validation errors
                        Object.keys(errorMsg.errors).forEach(function(field) {
                            var fieldId = '#onboarding_' + field;
                            if ($(fieldId).length) {
                                showFieldError(fieldId, errorMsg.errors[field][0]);
                            }
                        });
                    } else {
                        showNotice(errorMsg, 'error');
                    }
                    submitButton.prop('disabled', false).text(originalText);
                }
            },
            error: function(xhr, status, error) {
                $stepContent.removeClass('loading');
                var errorMsg = 'Account creation failed';
                if (xhr.responseJSON && xhr.responseJSON.data) {
                    errorMsg = xhr.responseJSON.data;
                }
                showNotice(errorMsg, 'error');
                submitButton.prop('disabled', false).text(originalText);
            }
        });
    }
    
    function showFieldError(fieldId, message) {
        var $field = $(fieldId);
        $field.addClass('error');
        $field.after('<span class="error-message">' + message + '</span>');
    }
    
    function isValidEmail(email) {
        var re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return re.test(email);
    }
    
    // Onboarding object
    window.brightTallyOnboarding = {
        currentStep: function() {
            var urlParams = new URLSearchParams(window.location.search);
            var step = parseInt(urlParams.get('step')) || 1;
            
            // If connected and on step 1, move to step 2
            if (step === 1) {
                $.ajax({
                    url: bright_tally_ajax.ajax_url,
                    type: 'POST',
                    async: false,
                    data: {
                        action: 'bright_tally_get_connection_status',
                        nonce: bright_tally_ajax.oauth_nonce
                    },
                    success: function(response) {
                        if (response.success && response.data.connected) {
                            step = 2;
                        }
                    }
                });
            }
            
            return step;
        },
        
        nextStep: function(step) {
            var current = this.currentStep();
            if (step) {
                current = step;
            } else {
                current++;
            }
            
            // Update step indicators
            $('.bright-tally-step-indicator .step').each(function(index) {
                var stepNum = index + 1;
                $(this).removeClass('active');
                if (stepNum < current) {
                    $(this).addClass('completed');
                } else if (stepNum === current) {
                    $(this).addClass('active');
                }
            });
            
            // Smooth transition
            var $onboarding = $('#bright-tally-onboarding');
            $onboarding.find('.bright-tally-onboarding-content').fadeOut(200, function() {
                window.location.href = bright_tally_admin.admin_url + '?page=bright-tally&step=' + current;
            });
        },
        
        previousStep: function() {
            var step = this.currentStep();
            if (step > 1) {
                step--;
                this.nextStep(step);
            }
        },
        
        complete: function() {
            var $onboarding = $('#bright-tally-onboarding');
            
            $.ajax({
                url: bright_tally_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'bright_tally_complete_onboarding',
                    nonce: bright_tally_ajax.oauth_nonce
                },
                success: function(response) {
                    if (response.success) {
                        $onboarding.fadeOut(300, function() {
                            $(this).remove();
                            // Redirect to polls page
                            window.location.href = bright_tally_admin.admin_url + '?page=bright-tally-polls';
                        });
                    } else {
                        showNotice('Failed to complete onboarding', 'error');
                    }
                },
                error: function() {
                    showNotice('Error completing onboarding', 'error');
                }
            });
        },
        
        skip: function() {
            if (confirm('Are you sure you want to skip the setup? You can always access these options later.')) {
                var $onboarding = $('#bright-tally-onboarding');
                
                $.ajax({
                    url: bright_tally_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'bright_tally_skip_onboarding',
                        nonce: bright_tally_ajax.oauth_nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            $onboarding.fadeOut(300, function() {
                                $(this).remove();
                            });
                        } else {
                            showNotice('Failed to skip onboarding', 'error');
                        }
                    },
                    error: function() {
                        showNotice('Error skipping onboarding', 'error');
                    }
                });
            }
        }
    };
    
    function testConnection() {
        var button = $('#test-connection');
        var result = $('#connection-result');
        var originalText = button.text();
        
        // Disable button and show loading
        button.prop('disabled', true).text('Testing...');
        result.html('');
        
        // Make AJAX request
        $.ajax({
            url: bright_tally_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'bright_tally_test_connection',
                nonce: bright_tally_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    result.html('<div class="notice notice-success"><p>' + response.data + '</p></div>');
                } else {
                    result.html('<div class="notice notice-error"><p>' + response.data + '</p></div>');
                }
            },
            error: function(xhr, status, error) {
                result.html('<div class="notice notice-error"><p>Connection test failed: ' + error + '</p></div>');
            },
            complete: function() {
                // Re-enable button
                button.prop('disabled', false).text(originalText);
            }
        });
    }
    
    function showShortcodeModal(pollId, pollSlug) {
        pollSlug = pollSlug || pollId;
        var shortcode = '[bright_tally id="' + pollSlug + '"]';
        $('#shortcode-input').val(shortcode);
        
        // Reset checkboxes
        $('#show-results').prop('checked', false);
        $('#dark-theme').prop('checked', false);
        $('#no-voting').prop('checked', false);
        
        // Show modal with animation
        $('#shortcode-modal').fadeIn(200);
        
        // Focus on input for easy selection
        setTimeout(function() {
            $('#shortcode-input').select();
        }, 300);
    }
    
    function updateShortcode() {
        var shortcodeInput = $('#shortcode-input');
        var pollId = shortcodeInput.val().match(/id="([^"]+)"/);
        
        if (!pollId) return;
        
        pollId = pollId[1];
        var shortcode = '[bright_tally id="' + pollId + '"';
        
        // Add options based on checkboxes
        if ($('#show-results').is(':checked')) {
            shortcode += ' show_results="true"';
        }
        if ($('#dark-theme').is(':checked')) {
            shortcode += ' theme="dark"';
        }
        if ($('#no-voting').is(':checked')) {
            shortcode += ' allow_voting="false"';
        }
        
        shortcode += ']';
        shortcodeInput.val(shortcode);
    }
    
    function copyToClipboard(element) {
        element.select();
        element[0].setSelectionRange(0, 99999); // For mobile devices
        
        try {
            var successful = document.execCommand('copy');
            if (successful) {
                var button = element.siblings('.copy-shortcode');
                var originalText = button.text();
                button.text('Copied!');
                setTimeout(function() {
                    button.text(originalText);
                }, 2000);
            }
        } catch (err) {
            console.error('Failed to copy text: ', err);
        }
    }
    
    // Utility functions
    function showNotice(message, type) {
        type = type || 'info';
        var noticeClass = 'notice-' + type;
        var notice = $('<div class="notice ' + noticeClass + ' is-dismissible"><p>' + message + '</p></div>');
        $('.wrap h1').after(notice);
        
        // Auto-dismiss after 5 seconds
        setTimeout(function() {
            notice.fadeOut();
        }, 5000);
    }
    
    function initiateQuickConnect() {
        var button = $('#quick-connect');
        var originalText = button.text();
        
        // Check if required objects exist
        if (typeof bright_tally_ajax === 'undefined') {
            console.error('BrightTally: bright_tally_ajax is not defined. JavaScript may not be loaded properly.');
            showNotice('Error: Plugin JavaScript not loaded. Please refresh the page.', 'error');
            return;
        }
        
        // Disable button and show loading
        button.prop('disabled', true).text('Connecting...');
        
        console.log('BrightTally: Initiating OAuth connection...');
        console.log('BrightTally: AJAX URL:', bright_tally_ajax.ajax_url);
        console.log('BrightTally: Action: bright_tally_get_oauth_url');
        
        // Get OAuth authorization URL
        $.ajax({
            url: bright_tally_ajax.ajax_url,
            type: 'POST',
            timeout: 30000, // 30 second timeout
            data: {
                action: 'bright_tally_get_oauth_url',
                nonce: bright_tally_ajax.oauth_nonce
            },
            beforeSend: function() {
                console.log('BrightTally: Sending AJAX request...');
            },
            success: function(response) {
                console.log('BrightTally: AJAX Success Response:', response);
                console.log('BrightTally: Response type:', typeof response);
                console.log('BrightTally: Response.success:', response.success);
                console.log('BrightTally: Response.data:', response.data);
                
                if (response.success && response.data && response.data.auth_url) {
                    console.log('BrightTally: Opening OAuth popup with URL:', response.data.auth_url);
                    console.log('BrightTally: Auth URL type check - is WordPress URL?', response.data.auth_url.indexOf('wp-admin') !== -1 || response.data.auth_url.indexOf('admin.php') !== -1);
                    console.log('BrightTally: Auth URL type check - is BrightTally URL?', response.data.auth_url.indexOf('brighttally') !== -1);
                    
                    // Verify this is a WordPress URL, not a BrightTally URL
                    // If user is already logged in, auth_url should be WordPress callback URL
                    if (response.data.auth_url.indexOf('wp-admin') === -1 && 
                        response.data.auth_url.indexOf('admin.php') === -1 &&
                        response.data.auth_url.indexOf('brighttally') !== -1) {
                        console.error('BrightTally: ERROR - auth_url appears to be a BrightTally URL instead of WordPress URL!');
                        console.error('BrightTally: This will cause the popup to show dashboard instead of processing callback');
                        showNotice('Error: Invalid callback URL received. Please try again.', 'error');
                        button.prop('disabled', false).text(originalText);
                        return;
                    }
                    
                    // Open OAuth popup
                    openOAuthPopup(response.data.auth_url, response.data.state);
                } else {
                    // Check if login is required
                    if (response.data && response.data.requires_login) {
                        console.log('BrightTally: Login required, opening login URL:', response.data.login_url);
                        // Open login page in popup
                        openOAuthPopup(response.data.login_url, null, true);
                    } else {
                        var errorMsg = 'Failed to initiate connection';
                        if (response.data) {
                            if (response.data.message) {
                                errorMsg = response.data.message;
                            } else if (typeof response.data === 'string') {
                                errorMsg = response.data;
                            }
                        }
                        console.error('BrightTally: Connection failed:', response);
                        showNotice(errorMsg, 'error');
                        button.prop('disabled', false).text(originalText);
                    }
                }
            },
            error: function(xhr, status, error) {
                console.error('BrightTally: AJAX Error:', {
                    status: status,
                    error: error,
                    responseText: xhr.responseText,
                    statusCode: xhr.status,
                    readyState: xhr.readyState
                });
                
                // Log full response for debugging
                if (xhr.responseText) {
                    console.error('BrightTally: Full response text:', xhr.responseText);
                    try {
                        var errorData = JSON.parse(xhr.responseText);
                        console.error('BrightTally: Parsed error data:', errorData);
                    } catch (e) {
                        console.error('BrightTally: Response is not JSON');
                    }
                }
                
                var errorMsg = 'Connection failed: ' + error;
                if (xhr.responseJSON && xhr.responseJSON.data) {
                    errorMsg = xhr.responseJSON.data;
                } else if (xhr.responseText) {
                    try {
                        var errorData = JSON.parse(xhr.responseText);
                        if (errorData.data) {
                            errorMsg = errorData.data;
                        }
                    } catch (e) {
                        // Not JSON, use response text
                        if (xhr.responseText.length < 200) {
                            errorMsg = xhr.responseText;
                        }
                    }
                }
                
                showNotice(errorMsg, 'error');
                button.prop('disabled', false).text(originalText);
            },
            complete: function(xhr, status) {
                console.log('BrightTally: AJAX Complete. Status:', status);
                console.log('BrightTally: XHR Status Code:', xhr.status);
            }
        });
    }
    
    function openOAuthPopup(url, state, isLogin) {
        console.log('BrightTally: Opening popup with URL:', url);
        
        if (!url) {
            console.error('BrightTally: No URL provided to openOAuthPopup');
            showNotice('Error: No authorization URL received. Please check your API configuration.', 'error');
            $('#quick-connect').prop('disabled', false).text('Sign In to BrightTally');
            return;
        }
        
        var width = 600;
        var height = 700;
        var left = (screen.width / 2) - (width / 2);
        var top = (screen.height / 2) - (height / 2);
        
        var popup = window.open(
            url,
            'brighttally_oauth',
            'width=' + width + ',height=' + height + ',left=' + left + ',top=' + top + ',toolbar=no,menubar=no,scrollbars=yes,resizable=yes'
        );
        
        if (!popup) {
            console.error('BrightTally: Popup blocked by browser');
            showNotice('Please allow popups for this site to connect your account', 'error');
            $('#quick-connect').prop('disabled', false).text('Sign In to BrightTally');
            return;
        }
        
        console.log('BrightTally: Popup opened successfully');
        
        // Declare pollTimer variable first (will be set below)
        var pollTimer;
        
        // Primary method: listen for message from popup
        // This is more reliable than polling, especially when user is already logged in
        // Edge case fix: When user is already logged in, popup might redirect to dashboard
        // before callback completes. postMessage ensures we catch the callback even if redirect happens.
        var messageListener = function(event) {
            // Verify message origin for security (WordPress admin URL)
            var adminUrl = new URL(bright_tally_ajax.ajax_url);
            var eventOrigin = event.origin || (event.source && event.source.location && event.source.location.origin);
            
            // Allow same origin or WordPress admin origin
            if (event.data && event.data.type === 'brighttally_oauth_callback') {
                if (pollTimer) {
                    clearInterval(pollTimer);
                }
                window.removeEventListener('message', messageListener);
                
                if (!popup.closed) {
                    popup.close();
                }
                
                if (event.data.success && event.data.code) {
                    console.log('BrightTally: Received OAuth callback message with code');
                    handleOAuthCallback(event.data.code, event.data.state);
                } else if (event.data.error) {
                    console.error('BrightTally: OAuth callback error:', event.data.error);
                    showNotice(event.data.error, 'error');
                    $('#quick-connect').prop('disabled', false).text('Sign In to BrightTally');
                } else {
                    // If popup closed without message, check status
                    console.log('BrightTally: Popup closed, checking connection status');
                    checkConnectionStatus();
                }
            }
        };
        window.addEventListener('message', messageListener);
        
        // Fallback: Monitor popup for callback (for cases where postMessage doesn't work)
        pollTimer = setInterval(function() {
            try {
                if (popup.closed) {
                    clearInterval(pollTimer);
                    window.removeEventListener('message', messageListener);
                    // Only check status if we didn't already handle the callback
                    setTimeout(function() {
                        checkConnectionStatus();
                    }, 500);
                    return;
                }
                
                // Check if popup has redirected to callback
                // Also check for wordpress_oauth flag (added to prevent redirects)
                var popupUrl = popup.location.href;
                if (popupUrl && (popupUrl.indexOf('oauth_callback') !== -1 || popupUrl.indexOf('wordpress_oauth') !== -1)) {
                    // Extract code from URL
                    var urlParams = new URLSearchParams(popupUrl.split('?')[1]);
                    var code = urlParams.get('code');
                    var stateParam = urlParams.get('state');
                    
                    if (code) {
                        clearInterval(pollTimer);
                        window.removeEventListener('message', messageListener);
                        if (!popup.closed) {
                            popup.close();
                        }
                        handleOAuthCallback(code, stateParam);
                    }
                }
            } catch (e) {
                // Cross-origin error is expected until redirect happens
                // This is normal when popup is on a different domain
            }
        }, 500);
        
        // Timeout after 5 minutes
        setTimeout(function() {
            if (!popup.closed) {
                clearInterval(pollTimer);
                popup.close();
                showNotice('Connection timeout. Please try again.', 'error');
            }
        }, 300000);
    }
    
    function handleOAuthCallback(code, state) {
        // Token is already stored by the callback page, just refresh status and reload
        showNotice('Account connected successfully!', 'success');
        checkConnectionStatus();
        // Reload page after short delay to show updated UI
        setTimeout(function() {
            window.location.reload();
        }, 1000);
    }
    
    function checkConnectionStatus() {
        $.ajax({
            url: bright_tally_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'bright_tally_get_connection_status',
                nonce: bright_tally_ajax.oauth_nonce
            },
            success: function(response) {
                if (response.success) {
                    updateConnectionUI(response.data);
                }
            }
        });
    }
    
    function updateConnectionUI(data) {
        var connectionStatus = $('.bright-tally-connection-status');
        var statusContent = $('.bright-tally-status-content');
        
        console.warn('updateConnectionUI', data);
        
        // Check if enhanced UI already exists (server-rendered with first-time message, next steps, etc.)
        var hasEnhancedUI = statusContent.find('.bright-tally-first-connection').length > 0 || 
                           statusContent.find('.bright-tally-next-steps').length > 0 ||
                           statusContent.find('.status-info').length > 0;
        
        if (hasEnhancedUI) {
            // Enhanced UI exists - only update the status classes and indicators
            // Don't overwrite the content since it has first-time messages and next steps
            if (data.connected) {
                connectionStatus.removeClass('disconnected').addClass('connected');
                // Update status indicator if it exists
                statusContent.find('.status-indicator').removeClass('disconnected').addClass('connected');
            } else {
                connectionStatus.removeClass('connected').addClass('disconnected');
                statusContent.find('.status-indicator').removeClass('connected').addClass('disconnected');
            }
        } else {
            // Simple fallback UI - update everything (for backwards compatibility)
            if (data.connected) {
                connectionStatus.removeClass('disconnected').addClass('connected')
                    .html('<span class="status-indicator connected"></span> Connected as ' + 
                        (data.user ? data.user.email : '') + ' at <strong>' + (data.api_url ? data.api_url : '') + '</strong>.');
            } else {
                connectionStatus.removeClass('connected').addClass('disconnected')
                    .html('<span class="status-indicator disconnected"></span> Not connected to BrightTally.');
            }
        }
    }
    
    function showEmailLoginForm() {
        $('#email-login-form').slideDown(300);
        $('html, body').animate({
            scrollTop: $('#email-login-form').offset().top - 100
        }, 500);
    }
    
    function showCreateAccountForm() {
        $('#create-account-form').slideDown(300);
        $('html, body').animate({
            scrollTop: $('#create-account-form').offset().top - 100
        }, 500);
    }
    
    function loginWithEmail() {
        var form = $('#email-login-account-form');
        var email = form.find('input[name="email"]').val();
        var password = form.find('input[name="password"]').val();
        var errorDiv = $('#email-login-error');
        
        // Clear previous errors
        errorDiv.hide().text('');
        
        // Validation
        if (!email || !password) {
            errorDiv.text('Please fill in all fields').show();
            return;
        }
        
        var submitButton = form.find('button[type="submit"]');
        var originalText = submitButton.text();
        submitButton.prop('disabled', true).text('Signing In...');
        
        $.ajax({
            url: bright_tally_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'bright_tally_login',
                email: email,
                password: password,
                nonce: bright_tally_ajax.create_account_nonce
            },
            success: function(response) {
                if (response.success) {
                    showNotice(response.data.message || 'Account connected successfully!', 'success');
                    // Reload page after short delay
                    setTimeout(function() {
                        window.location.reload();
                    }, 1500);
                } else {
                    var errorMsg = response.data || 'Login failed. Please check your credentials.';
                    errorDiv.text(errorMsg).show();
                    submitButton.prop('disabled', false).text(originalText);
                }
            },
            error: function(xhr, status, error) {
                var errorMsg = 'Login failed: ' + error;
                if (xhr.responseJSON && xhr.responseJSON.data) {
                    errorMsg = xhr.responseJSON.data;
                }
                errorDiv.text(errorMsg).show();
                submitButton.prop('disabled', false).text(originalText);
            }
        });
    }
    
    function createAccount() {
        var form = $('#create-account-form').find('form');
        var name = form.find('input[name="name"]').val();
        var email = form.find('input[name="email"]').val();
        var password = form.find('input[name="password"]').val();
        var passwordConfirm = form.find('input[name="password_confirm"]').val();
        
        // Validation
        if (!name || !email || !password || !passwordConfirm) {
            showNotice('Please fill in all fields', 'error');
            return;
        }
        
        if (password !== passwordConfirm) {
            showNotice('Passwords do not match', 'error');
            return;
        }
        
        if (password.length < 8) {
            showNotice('Password must be at least 8 characters long', 'error');
            return;
        }
        
        var submitButton = form.find('button[type="submit"]');
        var originalText = submitButton.text();
        submitButton.prop('disabled', true).text('Creating Account...');
        
        $.ajax({
            url: bright_tally_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'bright_tally_create_account',
                name: name,
                email: email,
                password: password,
                nonce: bright_tally_ajax.create_account_nonce
            },
            success: function(response) {
                if (response.success) {
                    showNotice(response.data.message || 'Account created and connected successfully!', 'success');
                    // Reload page after short delay
                    setTimeout(function() {
                        window.location.reload();
                    }, 1500);
                } else {
                    showNotice(response.data || 'Account creation failed', 'error');
                    submitButton.prop('disabled', false).text(originalText);
                }
            },
            error: function(xhr, status, error) {
                showNotice('Account creation failed: ' + error, 'error');
                submitButton.prop('disabled', false).text(originalText);
            }
        });
    }
    
    function disconnectAccount() {
        var message = 'Disconnecting removes BrightTally access from WordPress until you reconnect. Existing embeds keep working using their last synced settings.\n\nIMPORTANT: This will NOT close or delete your BrightTally account. Your account, polls, and data remain safe on brighttally.com.\n\nContinue?';
        if (!confirm(message)) {
            return;
        }
        
        $.ajax({
            url: bright_tally_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'bright_tally_disconnect',
                nonce: bright_tally_ajax.oauth_nonce
            },
            success: function(response) {
                if (response.success) {
                    showNotice(response.data.message || 'Account disconnected successfully', 'success');
                    setTimeout(function() {
                        window.location.reload();
                    }, 1500);
                } else {
                    showNotice(response.data || 'Disconnection failed', 'error');
                }
            },
            error: function(xhr, status, error) {
                showNotice('Disconnection failed: ' + error, 'error');
            }
        });
    }
    
    function showManualConfig() {
        $('#manual-config').slideDown(300);
        $('html, body').animate({
            scrollTop: $('#manual-config').offset().top - 100
        }, 500);
    }
    
    function getDocsBaseUrl() {
        if (typeof bright_tally_ajax !== 'undefined' && bright_tally_ajax.docs_url) {
            return bright_tally_ajax.docs_url;
        }
        return 'https://brighttally.com/docs';
    }
    
    // Expose functions globally if needed
    window.BrightTallyAdmin = {
        showNotice: showNotice,
        testConnection: testConnection,
        copyToClipboard: copyToClipboard,
        initiateQuickConnect: initiateQuickConnect,
        showManualConfig: showManualConfig,
        checkConnectionStatus: checkConnectionStatus,
        createAccount: createAccount,
        disconnectAccount: disconnectAccount,
        refreshPolls: refreshPolls,
        showShortcodeModal: showShortcodeModal
    };
    
})(jQuery);
