<?php
/**
 * Admin settings display
 *
 * @package Bright_Tally
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Handle form submission
if (isset($_POST['submit']) && wp_verify_nonce($_POST['bright_tally_settings_nonce'], 'bright_tally_settings')) {
    update_option('bright_tally_api_key', sanitize_text_field($_POST['bright_tally_api_key']));
    update_option('bright_tally_api_url', esc_url_raw($_POST['bright_tally_api_url']));
    update_option('bright_tally_default_theme', sanitize_text_field($_POST['bright_tally_default_theme']));
    update_option('bright_tally_cache_duration', intval($_POST['bright_tally_cache_duration']));
    
    echo '<div class="notice notice-success"><p>' . __('Settings saved successfully!', 'bright-tally') . '</p></div>';
}

// Get current settings
$api_key = get_option('bright_tally_api_key', '');
// Use URL helper to get current API URL (respects environment detection)
$api_url = Bright_Tally_URL_Helper::get_api_url();
$default_theme = get_option('bright_tally_default_theme', 'light');
$cache_duration = get_option('bright_tally_cache_duration', 300);
$docs_url_option = get_option('bright_tally_docs_url', '');
$docs_url_effective = Bright_Tally_URL_Helper::get_docs_url();

// Get connection status
$is_connected = Bright_Tally_Token_Manager::is_connected();
$user_data = Bright_Tally_Token_Manager::get_user_data();
$connection_status = Bright_Tally_Token_Manager::get_connection_status();

// Check if this is a first-time connection
// First-time means: user just connected AND we haven't shown the welcome message yet
$is_first_time = false;
$first_time_key = 'bright_tally_first_connection_shown_' . ($user_data['id'] ?? 'default');

// Check if connection was just made (either via URL parameter or by checking if welcome was shown)
if ($is_connected && $user_data) {
    // Check if we've already shown the first-time welcome message for this connection
    $first_time_shown = get_transient($first_time_key);
    
    // If not shown yet, and either:
    // 1. URL has connected=1 parameter (fresh connection), OR
    // 2. The connection was just made (we just set connected status)
    if (!$first_time_shown) {
        // Check if this page load is right after connection
        $just_connected = isset($_GET['connected']) && $_GET['connected'] == '1';
        
        if ($just_connected) {
            $is_first_time = true;
            // Mark as shown (expires in 24 hours so it shows if they reconnect later)
            set_transient($first_time_key, true, DAY_IN_SECONDS);
        }
    }
}

// Handle OAuth callback
// This must be processed BEFORE any other logic to prevent redirects
// Edge case: If user is already logged in to BrightTally, they might get redirected
// to dashboard instead of completing the OAuth flow. This early exit prevents that.
// CRITICAL: This check must happen BEFORE any WordPress hooks or redirects can fire
if (isset($_GET['oauth_callback']) && isset($_GET['code']) && isset($_GET['wordpress_oauth'])) {
    // Check user capability
    if (!current_user_can('manage_options')) {
        wp_die('Unauthorized');
    }
    
    // Prevent any redirects by processing immediately
    // Don't let WordPress or other plugins interfere
    $code = sanitize_text_field($_GET['code']);
    $state = isset($_GET['state']) ? sanitize_text_field($_GET['state']) : '';
    
    // Validate code format
    if (!preg_match('/^[a-zA-Z0-9_-]+$/', $code)) {
        $oauth_error = 'Invalid authorization code format';
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <title>Connection Error</title>
            <meta charset="UTF-8">
        </head>
        <body>
            <p>Connection failed. Closing window...</p>
            <script>
            (function() {
                if (window.opener) {
                    window.opener.postMessage({
                        type: 'brighttally_oauth_callback',
                        error: '<?php echo esc_js($oauth_error); ?>',
                        success: false
                    }, '*');
                    window.close();
                } else {
                    window.location.href = '<?php echo esc_url(admin_url('admin.php?page=bright-tally-settings&oauth_error=1')); ?>';
                }
            })();
            </script>
        </body>
        </html>
        <?php
        exit;
    }
    
    // Validate state parameter for CSRF protection
    if (!empty($state)) {
        $stored_state = get_option('bright_tally_oauth_state', '');
        if (empty($stored_state) || !hash_equals($stored_state, $state)) {
            $oauth_error = 'Invalid state parameter';
            ?>
            <!DOCTYPE html>
            <html>
            <head>
                <title>Connection Error</title>
                <meta charset="UTF-8">
            </head>
            <body>
                <p>Connection failed. Closing window...</p>
                <script>
                (function() {
                    if (window.opener) {
                        window.opener.postMessage({
                            type: 'brighttally_oauth_callback',
                            error: '<?php echo esc_js($oauth_error); ?>',
                            success: false
                        }, '*');
                        window.close();
                    } else {
                        window.location.href = '<?php echo esc_url(admin_url('admin.php?page=bright-tally-settings&oauth_error=1')); ?>';
                    }
                })();
                </script>
            </body>
            </html>
            <?php
            exit;
        }
        // Clear state after use (one-time use)
        delete_option('bright_tally_oauth_state');
    }
    
    // Exchange code for token
    $redirect_uri = admin_url('admin.php?page=bright-tally-settings&oauth_callback=1');
    $oauth = new Bright_Tally_OAuth();
    $result = $oauth->exchange_code_for_token($code, $redirect_uri);
    
    if (!is_wp_error($result) && isset($result['access_token'])) {
        // Store token and user data
        Bright_Tally_Token_Manager::store_access_token($result['access_token']);
        
        if (isset($result['expires_in'])) {
            Bright_Tally_Token_Manager::store_token_expires_at(time() + intval($result['expires_in']));
        }
        
        if (isset($result['user'])) {
            Bright_Tally_Token_Manager::store_user_data($result['user']);
        }
        
        // Check if this is a first-time connection before updating status
        $previous_status = Bright_Tally_Token_Manager::get_connection_status();
        $is_first_time_connection = ($previous_status !== 'connected');
        
        Bright_Tally_Token_Manager::store_connection_status('connected');
        
        // Mark this as a first-time connection for the welcome message
        if ($is_first_time_connection) {
            // Clear any previous "shown" flag so the message appears
            delete_transient('bright_tally_first_connection_shown');
        }
        
        // Output minimal HTML and JavaScript to close popup immediately
        // This prevents any redirects or page loads that might interfere
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <title>Connecting...</title>
            <meta charset="UTF-8">
        </head>
        <body>
            <p>Connection successful! Closing window...</p>
            <script>
            // Immediately send message to parent and close
            // Don't wait for page load - this prevents redirects
            (function() {
                if (window.opener) {
                    window.opener.postMessage({
                        type: 'brighttally_oauth_callback',
                        code: '<?php echo esc_js($code); ?>',
                        state: '<?php echo esc_js($state); ?>',
                        success: true
                    }, '*');
                    // Close immediately
                    window.close();
                } else {
                    // If not in popup, redirect to settings page
                    window.location.href = '<?php echo esc_url(admin_url('admin.php?page=bright-tally-settings&connected=1')); ?>';
                }
            })();
            </script>
        </body>
        </html>
        <?php
        exit;
    } else {
        // Don't expose detailed error messages to prevent information disclosure
        $oauth_error = 'Connection failed. Please try again.';
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <title>Connection Error</title>
            <meta charset="UTF-8">
        </head>
        <body>
            <p>Connection failed. Closing window...</p>
            <script>
            (function() {
                if (window.opener) {
                    window.opener.postMessage({
                        type: 'brighttally_oauth_callback',
                        error: '<?php echo esc_js($oauth_error); ?>',
                        success: false
                    }, '*');
                    window.close();
                } else {
                    // If not in popup, redirect to settings page with error
                    window.location.href = '<?php echo esc_url(admin_url('admin.php?page=bright-tally-settings&oauth_error=1')); ?>';
                }
            })();
            </script>
        </body>
        </html>
        <?php
        exit;
    }
}

// Show success/error messages
if (isset($_GET['connected']) && $_GET['connected'] === '1') {
    echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Account connected successfully!', 'bright-tally') . '</p></div>';
}
if (isset($_GET['oauth_error']) && $_GET['oauth_error'] === '1') {
    echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('OAuth connection failed. Please try again.', 'bright-tally') . '</p></div>';
}
?>

<div class="bright-tally-admin-page">
    <div class="bright-tally-admin-content">
        <div class="bright-tally-header">
            <div class="bright-tally-header-content">
                <div class="bright-tally-logo">
                    <svg viewBox="0 0 24 24" fill="currentColor">
                        <path d="M12 2L2 7l10 5 10-5-10-5zM2 17l10 5 10-5M2 12l10 5 10-5"/>
                    </svg>
                </div>
                <h1><?php _e('Account Settings', 'bright-tally'); ?></h1>
                <p><?php _e('Connect your BrightTally account to start creating and managing polls.', 'bright-tally'); ?></p>
                <div class="bright-tally-header-actions" style="margin-top: 15px;">
                    <a href="<?php echo esc_url($docs_url_effective . '/wordpress'); ?>" target="_blank" rel="noopener noreferrer" class="button button-secondary">
                        <svg viewBox="0 0 24 24" fill="currentColor" style="width: 16px; height: 16px; margin-right: 8px; vertical-align: middle;">
                            <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-6h2v6zm0-8h-2V7h2v2z"/>
                        </svg>
                        <?php _e('View Documentation', 'bright-tally'); ?>
                    </a>
                    <a href="<?php echo esc_url($docs_url_effective . '/wordpress/setup/connecting-the-wordpress-plugin'); ?>" target="_blank" rel="noopener noreferrer" class="button button-secondary">
                        <?php _e('Connection Help', 'bright-tally'); ?>
                    </a>
                </div>
            </div>
        </div>
        
        <div class="bright-tally-settings-content">
    
    <!-- Connection Status -->
    <div class="bright-tally-connection-status-section">
        <div class="bright-tally-status-card">
            <div class="bright-tally-status-header">
                <h3><?php _e('Connection Status', 'bright-tally'); ?></h3>
            </div>
            <div class="bright-tally-status-content">
                <div class="bright-tally-connection-status <?php echo $is_connected ? 'connected' : 'disconnected'; ?>">
                    <?php if ($is_connected && $user_data): ?>
                        <?php if ($is_first_time): ?>
                            <!-- First-Time Connection Congratulations -->
                            <div class="bright-tally-first-connection" style="background: linear-gradient(135deg, #10B981 0%, #059669 100%); color: white; padding: 24px; border-radius: 8px; margin-bottom: 20px; box-shadow: 0 4px 6px rgba(0,0,0,0.1);">
                                <div style="display: flex; align-items: center; margin-bottom: 16px;">
                                    <svg style="width: 32px; height: 32px; margin-right: 12px;" fill="currentColor" viewBox="0 0 20 20">
                                        <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"/>
                                    </svg>
                                    <h2 style="margin: 0; font-size: 24px; font-weight: bold;"><?php _e('Congratulations! You\'re Connected!', 'bright-tally'); ?></h2>
                                </div>
                                <p style="font-size: 16px; margin: 0 0 20px 0; opacity: 0.95;">
                                    <?php printf(__('Your BrightTally account (%s) is now connected to WordPress. You\'re all set to start creating amazing polls and forms!', 'bright-tally'), esc_html($user_data['email'] ?? '')); ?>
                                </p>
                                <div style="display: flex; gap: 12px; flex-wrap: wrap;">
                                    <a href="https://brighttally.com/dashboard/polls/create" target="_blank" rel="noopener noreferrer" class="button button-primary" style="background: white; color: #059669; border-color: white; font-weight: bold;">
                                        <?php _e('🎉 Create Your First Poll', 'bright-tally'); ?>
                                    </a>
                                    <a href="https://brighttally.com/docs/wordpress/manage/embedding-polls-and-forms" target="_blank" rel="noopener noreferrer" class="button" style="background: rgba(255,255,255,0.2); color: white; border-color: rgba(255,255,255,0.3);">
                                        <?php _e('📚 Learn How to Embed', 'bright-tally'); ?>
                                    </a>
                                </div>
                            </div>
                        <?php endif; ?>
                        
                        <span class="status-indicator connected"></span>
                        <div class="status-info">
                            <strong><?php _e('Connected', 'bright-tally'); ?></strong>
                            <p style="margin: 8px 0 4px 0;">
                                <?php printf(__('Connected as <strong>%s</strong>', 'bright-tally'), esc_html($user_data['email'] ?? '')); ?>
                            </p>
                            <p style="margin: 4px 0; color: #6B7280; font-size: 13px;">
                                <?php printf(__('API: %s', 'bright-tally'), esc_html($api_url ?? '')); ?>
                            </p>
                        </div>
                        
                        <!-- Next Steps -->
                        <div class="bright-tally-next-steps" style="margin-top: 20px; padding: 20px; background: #F9FAFB; border-radius: 8px; border-left: 4px solid #3B82F6;">
                            <h4 style="margin: 0 0 12px 0; font-size: 16px; font-weight: 600; color: #111827;">
                                <?php _e('Next Steps:', 'bright-tally'); ?>
                            </h4>
                            <ul style="margin: 0; padding-left: 20px; color: #374151;">
                                <li style="margin-bottom: 8px;">
                                    <a href="https://brighttally.com/dashboard/polls/create" target="_blank" rel="noopener noreferrer" style="color: #3B82F6; text-decoration: none; font-weight: 500;">
                                        <?php _e('Create a poll or form', 'bright-tally'); ?>
                                    </a>
                                    <?php _e('on BrightTally.com', 'bright-tally'); ?>
                                </li>
                                <li style="margin-bottom: 8px;">
                                    <a href="<?php echo esc_url($docs_url_effective . '/wordpress/manage/embedding-polls-and-forms'); ?>" target="_blank" rel="noopener noreferrer" style="color: #3B82F6; text-decoration: none; font-weight: 500;">
                                        <?php _e('Learn how to embed', 'bright-tally'); ?>
                                    </a>
                                    <?php _e('polls in your WordPress content', 'bright-tally'); ?>
                                </li>
                                <li style="margin-bottom: 8px;">
                                    <a href="https://brighttally.com/dashboard" target="_blank" rel="noopener noreferrer" style="color: #3B82F6; text-decoration: none; font-weight: 500;">
                                        <?php _e('Manage your polls', 'bright-tally'); ?>
                                    </a>
                                    <?php _e('and view analytics', 'bright-tally'); ?>
                                </li>
                                <li>
                                    <a href="<?php echo esc_url(admin_url('admin.php?page=bright-tally')); ?>" style="color: #3B82F6; text-decoration: none; font-weight: 500;">
                                        <?php _e('Browse your library', 'bright-tally'); ?>
                                    </a>
                                    <?php _e('to see available polls', 'bright-tally'); ?>
                                </li>
                            </ul>
                        </div>
                        
                        <div class="status-actions" style="margin-top: 20px; padding-top: 20px; border-top: 1px solid #E5E7EB;">
                            <button type="button" class="button button-secondary bright-tally-disconnect" data-context="settings">
                                <?php _e('Disconnect', 'bright-tally'); ?>
                            </button>
                            <p class="disconnect-note" style="margin-top: 12px; font-size: 13px; color: #6B7280;">
                                <?php _e('Disconnecting will remove access to your BrightTally library in WordPress until you reconnect. Existing embeds will continue to render using cached data.', 'bright-tally'); ?>
                                <br>
                                <strong><?php _e('Important:', 'bright-tally'); ?></strong> <?php _e('Disconnecting or uninstalling this plugin will NOT close or delete your BrightTally account. Your account, polls, and data remain safe on brighttally.com.', 'bright-tally'); ?>
                            </p>
                        </div>
                    <?php else: ?>
                        <span class="status-indicator disconnected"></span>
                        <div class="status-info">
                            <strong><?php _e('Not Connected', 'bright-tally'); ?></strong>
                            <p><?php _e('Connect your account to start using BrightTally', 'bright-tally'); ?></p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
    
    <?php if (!$is_connected): ?>
    <!-- Account Connection Section -->
    <div class="bright-tally-connection-section">
        <div class="bright-tally-connection-card">
            <div class="bright-tally-connection-header">
                <div class="bright-tally-connection-icon">
                    <svg viewBox="0 0 24 24" fill="currentColor">
                        <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z"/>
                    </svg>
                </div>
                <div class="bright-tally-connection-info">
                    <h3><?php _e('Connect Your Account', 'bright-tally'); ?></h3>
                    <p><?php _e('Choose how you\'d like to connect your BrightTally account.', 'bright-tally'); ?></p>
                </div>
            </div>
            
            <div class="bright-tally-connection-options">
                <div class="bright-tally-connection-option">
                    <div class="bright-tally-option-icon">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z"/>
                        </svg>
                    </div>
                    <div class="bright-tally-option-content">
                        <h4><?php _e('Quick Connect', 'bright-tally'); ?></h4>
                        <p><?php _e('Sign in to your BrightTally account for automatic setup.', 'bright-tally'); ?></p>
                        <button type="button" class="button button-primary bright-tally-connect-btn" id="quick-connect">
                            <?php _e('Sign In to BrightTally', 'bright-tally'); ?>
                        </button>
                    </div>
                </div>
                
                <div class="bright-tally-connection-divider">
                    <span><?php _e('or', 'bright-tally'); ?></span>
                </div>
                
                <div class="bright-tally-connection-option">
                    <div class="bright-tally-option-icon">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm0 3c1.66 0 3 1.34 3 3s-1.34 3-3 3-3-1.34-3-3 1.34-3 3-3zm0 14.2c-2.5 0-4.71-1.28-6-3.22.03-1.99 4-3.08 6-3.08 1.99 0 5.97 1.09 6 3.08-1.29 1.94-3.5 3.22-6 3.22z"/>
                        </svg>
                    </div>
                    <div class="bright-tally-option-content">
                        <h4><?php _e('Sign In with Email', 'bright-tally'); ?></h4>
                        <p><?php _e('Already have an account? Sign in with your email and password.', 'bright-tally'); ?></p>
                        <button type="button" class="button bright-tally-login-btn" id="email-login-btn">
                            <?php _e('Sign In', 'bright-tally'); ?>
                        </button>
                    </div>
                </div>
                
                <div class="bright-tally-connection-divider">
                    <span><?php _e('or', 'bright-tally'); ?></span>
                </div>
                
                <div class="bright-tally-connection-option">
                    <div class="bright-tally-option-icon">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm5 11h-4v4h-2v-4H7v-2h4V7h2v4h4v2z"/>
                        </svg>
                    </div>
                    <div class="bright-tally-option-content">
                        <h4><?php _e('Create New Account', 'bright-tally'); ?></h4>
                        <p><?php _e('Don\'t have an account? Create one now and get started instantly.', 'bright-tally'); ?></p>
                        <a href="https://brighttally.com/wordpress/onboarding" target="_blank" class="button bright-tally-create-btn">
                            <?php _e('Create Account', 'bright-tally'); ?>
                        </a>
                    </div>
                </div>
                
                <div class="bright-tally-connection-divider">
                    <span><?php _e('or', 'bright-tally'); ?></span>
                </div>
                
                <div class="bright-tally-connection-option">
                    <div class="bright-tally-option-icon">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M12 1L3 5v6c0 5.55 3.84 10.74 9 12 5.16-1.26 9-6.45 9-12V5l-9-4z"/>
                        </svg>
                    </div>
                    <div class="bright-tally-option-content">
                        <h4><?php _e('Manual Setup (Advanced)', 'bright-tally'); ?></h4>
                        <p><?php _e('Use an API token for programmatic access. OAuth connection is recommended.', 'bright-tally'); ?></p>
                        <button type="button" class="button bright-tally-manual-btn" id="manual-setup">
                            <?php _e('Enter API Token', 'bright-tally'); ?>
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Email/Password Login Form -->
    <div class="bright-tally-email-login" id="email-login-form" style="display: none;">
        <div class="bright-tally-config-header">
            <h3><?php _e('Sign In to BrightTally', 'bright-tally'); ?></h3>
            <p><?php _e('Enter your BrightTally account credentials to connect.', 'bright-tally'); ?></p>
        </div>
        
        <form id="email-login-account-form">
            <div class="bright-tally-form-grid">
                <div class="bright-tally-form-group">
                    <label for="login_email"><?php _e('Email Address', 'bright-tally'); ?> <span class="required">*</span></label>
                    <input type="email" 
                           id="login_email" 
                           name="email" 
                           class="bright-tally-input" 
                           placeholder="<?php esc_attr_e('you@example.com', 'bright-tally'); ?>"
                           required>
                </div>
                
                <div class="bright-tally-form-group">
                    <label for="login_password"><?php _e('Password', 'bright-tally'); ?> <span class="required">*</span></label>
                    <input type="password" 
                           id="login_password" 
                           name="password" 
                           class="bright-tally-input" 
                           placeholder="<?php esc_attr_e('Enter your password', 'bright-tally'); ?>"
                           required>
                </div>
            </div>
            
            <div class="bright-tally-form-actions">
                <button type="submit" class="button button-primary">
                    <?php _e('Sign In & Connect', 'bright-tally'); ?>
                </button>
                <button type="button" class="button" onclick="jQuery('#email-login-form').slideUp(300);">
                    <?php _e('Cancel', 'bright-tally'); ?>
                </button>
            </div>
            
            <div id="email-login-error" class="bright-tally-error-message" style="display: none; margin-top: 15px;"></div>
        </form>
    </div>
    
    <!-- Create Account Form -->
    <div class="bright-tally-create-account" id="create-account-form" style="display: none;">
        <div class="bright-tally-config-header">
            <h3><?php _e('Create BrightTally Account', 'bright-tally'); ?></h3>
            <p><?php _e('Create a new account to get started with BrightTally. Your account will be automatically connected.', 'bright-tally'); ?></p>
        </div>
        
        <form>
            <div class="bright-tally-form-grid">
                <div class="bright-tally-form-group">
                    <label for="create_name"><?php _e('Full Name', 'bright-tally'); ?> <span class="required">*</span></label>
                    <input type="text" 
                           id="create_name" 
                           name="name" 
                           class="bright-tally-input" 
                           placeholder="<?php esc_attr_e('John Doe', 'bright-tally'); ?>"
                           required>
                </div>
                
                <div class="bright-tally-form-group">
                    <label for="create_email"><?php _e('Email Address', 'bright-tally'); ?> <span class="required">*</span></label>
                    <input type="email" 
                           id="create_email" 
                           name="email" 
                           class="bright-tally-input" 
                           placeholder="<?php esc_attr_e('you@example.com', 'bright-tally'); ?>"
                           required>
                </div>
                
                <div class="bright-tally-form-group">
                    <label for="create_password"><?php _e('Password', 'bright-tally'); ?> <span class="required">*</span></label>
                    <input type="password" 
                           id="create_password" 
                           name="password" 
                           class="bright-tally-input" 
                           placeholder="<?php esc_attr_e('At least 8 characters', 'bright-tally'); ?>"
                           minlength="8"
                           required>
                    <p class="bright-tally-description">
                        <?php _e('Password must be at least 8 characters long.', 'bright-tally'); ?>
                    </p>
                </div>
                
                <div class="bright-tally-form-group">
                    <label for="create_password_confirm"><?php _e('Confirm Password', 'bright-tally'); ?> <span class="required">*</span></label>
                    <input type="password" 
                           id="create_password_confirm" 
                           name="password_confirm" 
                           class="bright-tally-input" 
                           placeholder="<?php esc_attr_e('Confirm your password', 'bright-tally'); ?>"
                           required>
                </div>
            </div>
            
            <div class="bright-tally-form-actions">
                <button type="submit" class="button button-primary">
                    <?php _e('Create Account & Connect', 'bright-tally'); ?>
                </button>
                <button type="button" class="button" onclick="jQuery('#create-account-form').slideUp(300);">
                    <?php _e('Cancel', 'bright-tally'); ?>
                </button>
            </div>
        </form>
    </div>
    
    <?php endif; ?>

    <!-- Manual Configuration Form -->
    <div class="bright-tally-manual-config" id="manual-config" style="display: none;">
        <div class="bright-tally-config-header">
            <h3><?php _e('API Token Configuration (Advanced)', 'bright-tally'); ?></h3>
            <p><?php _e('Optional: Enter an API token for programmatic access. OAuth connection (above) is recommended and easier to set up.', 'bright-tally'); ?></p>
        </div>
        
        <form method="post" action="">
            <?php wp_nonce_field('bright_tally_settings', 'bright_tally_settings_nonce'); ?>
            
            <div class="bright-tally-form-grid">
                <div class="bright-tally-form-group">
                    <label for="bright_tally_api_key"><?php _e('API Token (Optional)', 'bright-tally'); ?></label>
                    <input type="text" 
                           id="bright_tally_api_key" 
                           name="bright_tally_api_key" 
                           value="<?php echo esc_attr($api_key); ?>" 
                           class="bright-tally-input" 
                           placeholder="1|xxxxxxxxxxxxxxxx">
                    <p class="bright-tally-description">
                        <?php _e('Optional: Enter an API token for programmatic access. This is not required if you use OAuth connection above.', 'bright-tally'); ?>
                        <br>
                        <small style="color: #6B7280;">
                            <strong><?php _e('Note:', 'bright-tally'); ?></strong> <?php _e('API tokens are only available for Pro, Agency, and Enterprise plans. OAuth connection (recommended above) works for all plans and is easier to set up.', 'bright-tally'); ?>
                        </small>
                    </p>
                </div>
                
                <div class="bright-tally-form-group">
                    <label for="bright_tally_api_url"><?php _e('API URL', 'bright-tally'); ?></label>
                    <input type="url" 
                           id="bright_tally_api_url" 
                           name="bright_tally_api_url" 
                           value="<?php echo esc_attr($api_url); ?>" 
                           class="bright-tally-input"
                           placeholder="https://brighttally.com">
                    <p class="bright-tally-description">
                        <?php _e('The base URL for the BrightTally API.', 'bright-tally'); ?>
                        <?php
                        $env = Bright_Tally_URL_Helper::get_environment();
                        $env_label = ucfirst($env);
                        $env_color = $env === 'development' ? '#10B981' : ($env === 'staging' ? '#F59E0B' : '#6B7280');
                        ?>
                        <br>
                        <strong style="color: <?php echo esc_attr($env_color); ?>;">
                            <?php printf(__('Current Environment: %s', 'bright-tally'), $env_label); ?>
                        </strong>
                        <?php if ($env === 'development'): ?>
                            <br>
                            <small style="color: #6B7280;">
                                <?php _e('Development mode detected. URL auto-configured for local testing.', 'bright-tally'); ?>
                                <?php if (defined('BRIGHT_TALLY_API_URL')): ?>
                                    <br><?php _e('(Using BRIGHT_TALLY_API_URL constant from wp-config.php)', 'bright-tally'); ?>
                                <?php endif; ?>
                            </small>
                        <?php endif; ?>
                    </p>
                </div>
                
                <div class="bright-tally-form-group">
                    <label for="bright_tally_docs_url"><?php _e('Documentation Base URL', 'bright-tally'); ?></label>
                    <input type="url"
                           id="bright_tally_docs_url"
                           name="bright_tally_docs_url"
                           value="<?php echo esc_attr($docs_url_option); ?>"
                           class="bright-tally-input"
                           placeholder="<?php echo esc_attr($docs_url_effective); ?>">
                    <p class="bright-tally-description">
                        <?php _e('Leave blank to use the default docs base that matches your BrightTally environment. Update this when testing staging docs or a custom knowledge base.', 'bright-tally'); ?>
                        <br>
                        <small style="color: #6B7280;">
                            <?php printf(__('Current docs URL in use: %s', 'bright-tally'), '<code>' . esc_html($docs_url_effective) . '</code>'); ?>
                        </small>
                    </p>
                </div>
                
                <div class="bright-tally-form-group">
                    <label for="bright_tally_default_theme"><?php _e('Default Theme (WordPress Only)', 'bright-tally'); ?></label>
                    <select id="bright_tally_default_theme" name="bright_tally_default_theme" class="bright-tally-select">
                        <option value="light" <?php selected($default_theme, 'light'); ?>><?php _e('Light', 'bright-tally'); ?></option>
                        <option value="dark" <?php selected($default_theme, 'dark'); ?>><?php _e('Dark', 'bright-tally'); ?></option>
                    </select>
                    <p class="bright-tally-description">
                        <?php _e('This setting only affects polls embedded via WordPress shortcode when no theme is specified. It does not set a default theme in your BrightTally account. To set a default theme for new polls created in BrightTally, visit your BrightTally team settings.', 'bright-tally'); ?>
                    </p>
                </div>
                
                <div class="bright-tally-form-group">
                    <label for="bright_tally_cache_duration"><?php _e('Cache Duration (seconds)', 'bright-tally'); ?></label>
                    <input type="number" 
                           id="bright_tally_cache_duration" 
                           name="bright_tally_cache_duration" 
                           value="<?php echo esc_attr($cache_duration); ?>" 
                           min="60" 
                           max="3600" 
                           class="bright-tally-input">
                    <p class="bright-tally-description">
                        <?php _e('How long to cache poll data (60-3600 seconds).', 'bright-tally'); ?>
                    </p>
                </div>
            </div>
        
        <div class="bright-tally-test-connection">
            <h3><?php _e('Test Connection', 'bright-tally'); ?></h3>
            <p><?php _e('Test your API connection to make sure everything is working correctly.', 'bright-tally'); ?></p>
            <button type="button" id="test-connection" class="button">
                <?php _e('Test Connection', 'bright-tally'); ?>
            </button>
            <div id="connection-result" style="margin-top: 10px;"></div>
        </div>
        
        <?php submit_button(); ?>
    </form>
</div>

<script>
jQuery(document).ready(function($) {
    $('#test-connection').on('click', function() {
        var button = $(this);
        var result = $('#connection-result');
        
        button.prop('disabled', true).text('<?php _e('Testing...', 'bright-tally'); ?>');
        result.html('');
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'bright_tally_test_connection',
                nonce: '<?php echo wp_create_nonce('bright_tally_test_connection'); ?>'
            },
            success: function(response) {
                if (response.success) {
                    result.html('<div class="notice notice-success"><p>' + response.data + '</p></div>');
                } else {
                    result.html('<div class="notice notice-error"><p>' + response.data + '</p></div>');
                }
            },
            error: function() {
                result.html('<div class="notice notice-error"><p><?php _e('Connection test failed.', 'bright-tally'); ?></p></div>');
            },
            complete: function() {
                button.prop('disabled', false).text('<?php _e('Test Connection', 'bright-tally'); ?>');
            }
        });
    });
});
</script>
