<?php
/**
 * Admin polls display
 *
 * @package Bright_Tally
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Get API instance
$api = new Bright_Tally_API();

// Check if API is configured
if (!$api->is_configured()) {
    echo '<div class="notice notice-warning"><p>' . __('Please configure your API settings first.', 'bright-tally') . ' <a href="' . admin_url('admin.php?page=bright-tally-settings') . '">' . __('Go to Settings', 'bright-tally') . '</a></p></div>';
    return;
}

// Get polls from API
$polls = $api->get_polls();

if (is_wp_error($polls)) {
    echo '<div class="notice notice-error"><p>' . __('Error loading polls: ', 'bright-tally') . $polls->get_error_message() . '</p></div>';
    return;
}
?>

<div class="bright-tally-admin-page">
    <div class="bright-tally-admin-content">
        <div class="bright-tally-header">
            <div class="bright-tally-header-content">
                <div class="bright-tally-logo">
                    <svg viewBox="0 0 24 24" fill="currentColor">
                        <path d="M12 2L2 7l10 5 10-5-10-5zM2 17l10 5 10-5M2 12l10 5 10-5"/>
                    </svg>
                </div>
                <h1><?php _e('My Polls', 'bright-tally'); ?></h1>
                <p><?php _e('Manage and embed your polls from your BrightTally account.', 'bright-tally'); ?></p>
                <div class="bright-tally-header-actions" style="margin-top: 15px; margin-bottom: 15px;">
                    <a href="<?php echo esc_url(Bright_Tally_URL_Helper::get_docs_url() . '/wordpress/manage/embedding-polls-and-forms'); ?>" target="_blank" rel="noopener noreferrer" class="button button-secondary">
                        <svg viewBox="0 0 24 24" fill="currentColor" style="width: 16px; height: 16px; margin-right: 8px; vertical-align: middle;">
                            <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-6h2v6zm0-8h-2V7h2v2z"/>
                        </svg>
                        <?php _e('Embedding Guide', 'bright-tally'); ?>
                    </a>
                    <a href="<?php echo esc_url(Bright_Tally_URL_Helper::get_docs_url() . '/wordpress'); ?>" target="_blank" rel="noopener noreferrer" class="button button-secondary">
                        <?php _e('Full Documentation', 'bright-tally'); ?>
                    </a>
                </div>
                <div class="bright-tally-notice" style="background: #fff3cd; border-left: 4px solid #ffc107; padding: 12px; margin-top: 15px; color: #856404;">
                    <strong><?php _e('Note:', 'bright-tally'); ?></strong> <?php _e('Polls are created on brighttally.com. Click "Create New Poll" below to open the poll builder. After creating a poll, click the "Refresh" button below or wait a moment for it to appear here automatically.', 'bright-tally'); ?>
                </div>
            </div>
        </div>
        
        <div class="bright-tally-polls-page">
            <!-- Polls Header with Search and Filters -->
            <div class="bright-tally-polls-toolbar">
                <div class="bright-tally-search-box">
                    <svg class="search-icon" viewBox="0 0 24 24" fill="currentColor">
                        <path d="M15.5 14h-.79l-.28-.27C15.41 12.59 16 11.11 16 9.5 16 5.91 13.09 3 9.5 3S3 5.91 3 9.5 5.91 16 9.5 16c1.61 0 3.09-.59 4.23-1.57l.27.28v.79l5 4.99L20.49 19l-4.99-5zm-6 0C7.01 14 5 11.99 5 9.5S7.01 5 9.5 5 14 7.01 14 9.5 11.99 14 9.5 14z"/>
                    </svg>
                    <input type="text" 
                           id="poll-search" 
                           class="bright-tally-search-input" 
                           placeholder="<?php esc_attr_e('Search polls...', 'bright-tally'); ?>">
                </div>
                
                <div class="bright-tally-filter-buttons">
                    <button class="bright-tally-filter-btn active" data-filter="all">
                        <?php _e('All', 'bright-tally'); ?>
                    </button>
                    <button class="bright-tally-filter-btn" data-filter="active">
                        <?php _e('Active', 'bright-tally'); ?>
                    </button>
                    <button class="bright-tally-filter-btn" data-filter="draft">
                        <?php _e('Draft', 'bright-tally'); ?>
                    </button>
                    <button class="bright-tally-filter-btn" data-filter="closed">
                        <?php _e('Closed', 'bright-tally'); ?>
                    </button>
                </div>
                
                <div class="bright-tally-toolbar-actions">
                    <button id="refresh-polls" class="button" title="<?php esc_attr_e('Refresh polls', 'bright-tally'); ?>">
                        <svg viewBox="0 0 24 24" fill="currentColor" style="width: 16px; height: 16px; margin-right: 8px; vertical-align: middle;">
                            <path d="M17.65 6.35C16.2 4.9 14.21 4 12 4c-4.42 0-7.99 3.58-7.99 8s3.57 8 7.99 8c3.73 0 6.84-2.55 7.73-6h-2.08c-.82 2.33-3.04 4-5.65 4-3.31 0-6-2.69-6-6s2.69-6 6-6c1.66 0 3.14.69 4.22 1.78L13 11h7V4l-2.35 2.35z"/>
                        </svg>
                        <?php _e('Refresh', 'bright-tally'); ?>
                    </button>
                    <a href="https://brighttally.com/dashboard/polls/create" target="_blank" class="button button-primary">
                        <svg viewBox="0 0 24 24" fill="currentColor" style="width: 16px; height: 16px; margin-right: 8px; vertical-align: middle;">
                            <path d="M19 13h-6v6h-2v-6H5v-2h6V5h2v6h6v2z"/>
                        </svg>
                        <?php _e('Create New Poll', 'bright-tally'); ?>
                    </a>
                </div>
            </div>
            
            <!-- Loading State -->
            <div class="bright-tally-polls-loading" id="polls-loading" style="display: none;">
                <div class="bright-tally-spinner-large"></div>
                <p><?php _e('Loading polls...', 'bright-tally'); ?></p>
            </div>
            
            <!-- Polls Grid -->
            <?php if (empty($polls) || !isset($polls['data']) || empty($polls['data'])): ?>
                <div class="bright-tally-no-polls">
                    <div class="no-polls-icon">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M19 3H5c-1.1 0-2 .9-2 2v14c0 1.1.9 2 2 2h14c1.1 0 2-.9 2-2V5c0-1.1-.9-2-2-2zm-5 14H7v-2h7v2zm3-4H7v-2h10v2zm0-4H7V7h10v2z"/>
                        </svg>
                    </div>
                    <h3><?php _e('No polls found', 'bright-tally'); ?></h3>
                    <p><?php _e('You don\'t have any polls yet. Create your first poll to get started!', 'bright-tally'); ?></p>
                    <a href="https://brighttally.com/dashboard/polls/create" target="_blank" class="button button-primary button-large">
                        <?php _e('Create Your First Poll', 'bright-tally'); ?>
                    </a>
                </div>
            <?php else: ?>
                <div class="bright-tally-polls-grid" id="polls-grid">
                    <?php foreach ($polls['data'] as $poll): ?>
                        <div class="bright-tally-poll-card" 
                             data-poll-id="<?php echo esc_attr($poll['id']); ?>"
                             data-status="<?php echo esc_attr(strtolower($poll['status'] ?? 'active')); ?>"
                             data-title="<?php echo esc_attr(strtolower($poll['title'])); ?>">
                            <div class="bright-tally-poll-card-header">
                                <div class="bright-tally-poll-status-badge status-<?php echo esc_attr(strtolower($poll['status'] ?? 'active')); ?>">
                                    <span class="status-dot"></span>
                                    <span class="status-text"><?php echo esc_html(ucfirst($poll['status'] ?? 'Active')); ?></span>
                                </div>
                            </div>
                            
                            <div class="bright-tally-poll-card-body">
                                <h3 class="bright-tally-poll-title"><?php echo esc_html($poll['title']); ?></h3>
                                
                                <?php if (!empty($poll['description'])): ?>
                                    <p class="bright-tally-poll-description">
                                        <?php echo esc_html(wp_trim_words($poll['description'], 15)); ?>
                                    </p>
                                <?php endif; ?>
                                
                                <div class="bright-tally-poll-stats">
                                    <div class="bright-tally-stat-item">
                                        <svg viewBox="0 0 24 24" fill="currentColor">
                                            <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z"/>
                                        </svg>
                                        <span class="stat-value"><?php echo esc_html(number_format($poll['response_count'] ?? 0)); ?></span>
                                        <span class="stat-label"><?php _e('Responses', 'bright-tally'); ?></span>
                                    </div>
                                    <div class="bright-tally-stat-item">
                                        <svg viewBox="0 0 24 24" fill="currentColor">
                                            <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm0 18c-4.41 0-8-3.59-8-8s3.59-8 8-8 8 3.59 8 8-3.59 8-8 8zm.31-8.86c-1.77-.45-2.34-.94-2.34-1.67 0-.84.79-1.43 2.1-1.43 1.38 0 1.9.66 1.94 1.64h1.71c-.05-1.34-.87-2.57-2.49-2.97V5H10.9v1.69c-1.51.32-2.72 1.3-2.72 2.81 0 1.79 1.49 2.69 3.66 3.21 1.95.46 2.34 1.15 2.34 1.87 0 .53-.39 1.39-2.1 1.39-1.6 0-2.23-.72-2.32-1.64H8.04c.1 1.7 1.36 2.66 2.86 2.97V19h2.34v-1.67c1.52-.29 2.72-1.16 2.72-2.83 0-1.57-1.19-2.49-3.65-3.03z"/>
                                        </svg>
                                        <span class="stat-value"><?php echo esc_html(date('M j', strtotime($poll['created_at']))); ?></span>
                                        <span class="stat-label"><?php _e('Created', 'bright-tally'); ?></span>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="bright-tally-poll-card-footer">
                                <button class="button button-primary get-shortcode" data-poll-id="<?php echo esc_attr($poll['id']); ?>" data-poll-slug="<?php echo esc_attr($poll['slug'] ?? ''); ?>">
                                    <?php _e('Get Shortcode', 'bright-tally'); ?>
                                </button>
                                <a href="<?php echo esc_url($poll['url'] ?? '#'); ?>" target="_blank" class="button button-secondary">
                                    <?php _e('View', 'bright-tally'); ?>
                                </a>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Shortcode Modal -->
<div id="shortcode-modal" class="bright-tally-modal" style="display: none;">
    <div class="bright-tally-modal-content">
        <div class="bright-tally-modal-header">
            <h3><?php _e('Poll Shortcode', 'bright-tally'); ?></h3>
            <button class="bright-tally-modal-close">&times;</button>
        </div>
        <div class="bright-tally-modal-body">
            <p><?php _e('Copy this shortcode to embed the poll in your posts or pages:', 'bright-tally'); ?></p>
            <div class="bright-tally-shortcode-container">
                <input type="text" id="shortcode-input" readonly class="widefat">
                <button class="button copy-shortcode"><?php _e('Copy', 'bright-tally'); ?></button>
            </div>
            
            <h4><?php _e('Customization Options:', 'bright-tally'); ?></h4>
            <div class="bright-tally-shortcode-options">
                <label>
                    <input type="checkbox" id="show-results"> <?php _e('Show Results', 'bright-tally'); ?>
                </label>
                <label>
                    <input type="checkbox" id="dark-theme"> <?php _e('Dark Theme', 'bright-tally'); ?>
                </label>
                <label>
                    <input type="checkbox" id="no-voting"> <?php _e('Disable Voting', 'bright-tally'); ?>
                </label>
            </div>
        </div>
    </div>
</div>

<script>
jQuery(document).ready(function($) {
    // Search functionality
    $('#poll-search').on('keyup', function() {
        var searchTerm = $(this).val().toLowerCase();
        filterPolls();
    });
    
    // Filter buttons
    $('.bright-tally-filter-btn').on('click', function() {
        $('.bright-tally-filter-btn').removeClass('active');
        $(this).addClass('active');
        filterPolls();
    });
    
    function filterPolls() {
        var searchTerm = $('#poll-search').val().toLowerCase();
        var filter = $('.bright-tally-filter-btn.active').data('filter');
        var visibleCount = 0;
        
        $('.bright-tally-poll-card').each(function() {
            var $card = $(this);
            var title = $card.data('title') || '';
            var status = $card.data('status') || '';
            var matchesSearch = title.indexOf(searchTerm) !== -1 || searchTerm === '';
            var matchesFilter = filter === 'all' || status === filter;
            
            if (matchesSearch && matchesFilter) {
                $card.fadeIn(200);
                visibleCount++;
            } else {
                $card.fadeOut(200);
            }
        });
        
        // Show no results message if needed
        if (visibleCount === 0) {
            if ($('#no-results-message').length === 0) {
                $('#polls-grid').after('<div id="no-results-message" class="bright-tally-no-results"><p><?php _e('No polls match your search criteria.', 'bright-tally'); ?></p></div>');
            }
        } else {
            $('#no-results-message').remove();
        }
    }
    
    // Get shortcode button
    $('.get-shortcode').on('click', function() {
        var pollId = $(this).data('poll-id');
        var pollSlug = $(this).data('poll-slug') || pollId;
        var shortcode = '[bright_tally id="' + pollSlug + '"]';
        $('#shortcode-input').val(shortcode);
        $('#shortcode-modal').show();
    });
    
    // Close modal
    $('.bright-tally-modal-close, .bright-tally-modal').on('click', function(e) {
        if (e.target === this) {
            $('#shortcode-modal').hide();
        }
    });
    
    // Copy shortcode
    $('.copy-shortcode').on('click', function() {
        $('#shortcode-input').select();
        document.execCommand('copy');
        $(this).text('<?php _e('Copied!', 'bright-tally'); ?>');
        setTimeout(() => {
            $(this).text('<?php _e('Copy', 'bright-tally'); ?>');
        }, 2000);
    });
    
    // Update shortcode based on options
    function updateShortcode() {
        var shortcodeMatch = $('#shortcode-input').val().match(/id="([^"]+)"/);
        if (!shortcodeMatch) return;
        
        var pollId = shortcodeMatch[1];
        var shortcode = '[bright_tally id="' + pollId + '"';
        
        if ($('#show-results').is(':checked')) {
            shortcode += ' show_results="true"';
        }
        if ($('#dark-theme').is(':checked')) {
            shortcode += ' theme="dark"';
        }
        if ($('#no-voting').is(':checked')) {
            shortcode += ' allow_voting="false"';
        }
        
        shortcode += ']';
        $('#shortcode-input').val(shortcode);
    }
    
    $('.bright-tally-shortcode-options input').on('change', updateShortcode);
});
</script>
