<?php
/**
 * The admin-specific functionality of the plugin
 *
 * @package Bright_Tally
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Bright_Tally_Admin {
    
    /**
     * The ID of this plugin
     *
     * @since 1.0.0
     * @var string
     */
    private $plugin_name;
    
    /**
     * The version of this plugin
     *
     * @since 1.0.0
     * @var string
     */
    private $version;
    
    /**
     * Initialize the class and set its properties
     *
     * @since 1.0.0
     * @param string $plugin_name The name of this plugin
     * @param string $version The version of this plugin
     */
    public function __construct($plugin_name, $version) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
    }
    
    /**
     * Register the stylesheets for the admin area
     *
     * @since 1.0.0
     */
    public function enqueue_styles() {
        wp_enqueue_style(
            $this->plugin_name,
            BRIGHT_TALLY_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            $this->version,
            'all'
        );
    }
    
    /**
     * Register the JavaScript for the admin area
     *
     * @since 1.0.0
     */
    public function enqueue_scripts($hook) {
        wp_enqueue_script(
            $this->plugin_name,
            BRIGHT_TALLY_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery'),
            $this->version,
            false
        );
        
        // Localize script for AJAX
        wp_localize_script($this->plugin_name, 'bright_tally_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('bright_tally_nonce'),
            'oauth_nonce' => wp_create_nonce('bright_tally_oauth'),
            'create_account_nonce' => wp_create_nonce('bright_tally_create_account'),
            'docs_url' => Bright_Tally_URL_Helper::get_docs_url(),
        ));
        
        // Add admin URL for onboarding
        wp_localize_script($this->plugin_name, 'bright_tally_admin', array(
            'admin_url' => admin_url('admin.php')
        ));
        
        // Enqueue block editor assets
        if (function_exists('register_block_type') && (in_array($hook, array('post.php', 'post-new.php')) || $hook === 'site-editor.php')) {
            // Block assets are handled by WordPress automatically via block.json
        }
    }
    
    /**
     * Add the admin menu
     *
     * @since 1.0.0
     */
    public function add_plugin_admin_menu() {
        add_menu_page(
            __('BrightTally', 'bright-tally'),
            __('BrightTally', 'bright-tally'),
            'manage_options',
            'bright-tally',
            array($this, 'display_plugin_admin_dashboard'),
            'dashicons-chart-bar',
            30
        );
        
        add_submenu_page(
            'bright-tally',
            __('Settings', 'bright-tally'),
            __('Settings', 'bright-tally'),
            'manage_options',
            'bright-tally-settings',
            array($this, 'display_plugin_admin_settings')
        );
        
        add_submenu_page(
            'bright-tally',
            __('Polls', 'bright-tally'),
            __('Polls', 'bright-tally'),
            'manage_options',
            'bright-tally-polls',
            array($this, 'display_plugin_admin_polls')
        );
        
        add_submenu_page(
            'bright-tally',
            __('Documentation', 'bright-tally'),
            __('Documentation', 'bright-tally'),
            'manage_options',
            'bright-tally-docs',
            array($this, 'display_plugin_admin_docs')
        );
    }
    
    /**
     * Display the admin dashboard
     *
     * @since 1.0.0
     */
    public function display_plugin_admin_dashboard() {
        // Check if we should show onboarding
        $show_onboarding = false;
        $onboarding_completed = get_option('bright_tally_onboarding_completed', false);
        
        // Show onboarding if:
        // 1. Not completed yet
        // 2. User is not connected
        // 3. User hasn't explicitly skipped it
        if (!$onboarding_completed && !Bright_Tally_Token_Manager::is_connected()) {
            $show_onboarding = true;
        }
        
        // Check for activation redirect
        if (get_transient('bright_tally_activation_redirect')) {
            delete_transient('bright_tally_activation_redirect');
            $show_onboarding = true;
        }
        
        if ($show_onboarding) {
            include_once BRIGHT_TALLY_PLUGIN_DIR . 'admin/partials/bright-tally-onboarding.php';
        }
        
        include_once BRIGHT_TALLY_PLUGIN_DIR . 'admin/partials/bright-tally-admin-display.php';
    }
    
    /**
     * Display the admin settings page
     *
     * @since 1.0.0
     */
    public function display_plugin_admin_settings() {
        include_once BRIGHT_TALLY_PLUGIN_DIR . 'admin/partials/bright-tally-admin-settings.php';
    }
    
    /**
     * Display the admin polls page
     *
     * @since 1.0.0
     */
    public function display_plugin_admin_polls() {
        include_once BRIGHT_TALLY_PLUGIN_DIR . 'admin/partials/bright-tally-admin-polls.php';
    }
    
    /**
     * Display the documentation page
     *
     * @since 1.0.0
     */
    public function display_plugin_admin_docs() {
        $docs_url = Bright_Tally_URL_Helper::get_docs_url();
        ?>
        <div class="bright-tally-admin-page">
            <div class="bright-tally-admin-content">
                <div class="bright-tally-header">
                    <div class="bright-tally-header-content">
                        <div class="bright-tally-logo">
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M12 2L2 7l10 5 10-5-10-5zM2 17l10 5 10-5M2 12l10 5 10-5"/>
                            </svg>
                        </div>
                        <h1><?php _e('Documentation', 'bright-tally'); ?></h1>
                        <p><?php _e('Complete guides and tutorials for using BrightTally with WordPress.', 'bright-tally'); ?></p>
                    </div>
                </div>
                
                <div class="bright-tally-docs-page" style="max-width: 1200px; margin: 0 auto;">
                    <div class="bright-tally-docs-grid" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 20px; margin-top: 30px;">
                        <div class="bright-tally-doc-card" style="background: #fff; border: 1px solid #ddd; border-radius: 8px; padding: 24px;">
                            <h3 style="margin-top: 0;"><?php _e('Getting Started', 'bright-tally'); ?></h3>
                            <p><?php _e('Learn how to install, configure, and connect your BrightTally account.', 'bright-tally'); ?></p>
                            <a href="<?php echo esc_url($docs_url . '/wordpress/setup'); ?>" target="_blank" class="button button-primary">
                                <?php _e('View Setup Guide', 'bright-tally'); ?>
                            </a>
                        </div>
                        
                        <div class="bright-tally-doc-card" style="background: #fff; border: 1px solid #ddd; border-radius: 8px; padding: 24px;">
                            <h3 style="margin-top: 0;"><?php _e('Embedding Polls', 'bright-tally'); ?></h3>
                            <p><?php _e('Step-by-step guide to embedding polls using shortcodes and the Gutenberg block.', 'bright-tally'); ?></p>
                            <a href="<?php echo esc_url($docs_url . '/wordpress/manage/embedding-polls-and-forms'); ?>" target="_blank" class="button button-primary">
                                <?php _e('View Embedding Guide', 'bright-tally'); ?>
                            </a>
                        </div>
                        
                        <div class="bright-tally-doc-card" style="background: #fff; border: 1px solid #ddd; border-radius: 8px; padding: 24px;">
                            <h3 style="margin-top: 0;"><?php _e('Troubleshooting', 'bright-tally'); ?></h3>
                            <p><?php _e('Common issues and solutions for connection problems, OAuth errors, and more.', 'bright-tally'); ?></p>
                            <a href="<?php echo esc_url($docs_url . '/wordpress/troubleshooting'); ?>" target="_blank" class="button button-primary">
                                <?php _e('View Troubleshooting', 'bright-tally'); ?>
                            </a>
                        </div>
                        
                        <div class="bright-tally-doc-card" style="background: #fff; border: 1px solid #ddd; border-radius: 8px; padding: 24px;">
                            <h3 style="margin-top: 0;"><?php _e('Shortcode Reference', 'bright-tally'); ?></h3>
                            <p><?php _e('Complete reference for all shortcode attributes and customization options.', 'bright-tally'); ?></p>
                            <a href="<?php echo esc_url($docs_url . '/wordpress/manage/embedding-polls-and-forms#shortcodes'); ?>" target="_blank" class="button button-primary">
                                <?php _e('View Shortcode Docs', 'bright-tally'); ?>
                            </a>
                        </div>
                        
                        <div class="bright-tally-doc-card" style="background: #fff; border: 1px solid #ddd; border-radius: 8px; padding: 24px;">
                            <h3 style="margin-top: 0;"><?php _e('Account & Plans', 'bright-tally'); ?></h3>
                            <p><?php _e('Learn about subscription plans, features, and account management.', 'bright-tally'); ?></p>
                            <a href="<?php echo esc_url($docs_url . '/account'); ?>" target="_blank" class="button button-primary">
                                <?php _e('View Account Docs', 'bright-tally'); ?>
                            </a>
                        </div>
                        
                        <div class="bright-tally-doc-card" style="background: #fff; border: 1px solid #ddd; border-radius: 8px; padding: 24px;">
                            <h3 style="margin-top: 0;"><?php _e('Full Documentation', 'bright-tally'); ?></h3>
                            <p><?php _e('Browse all documentation, guides, and tutorials for BrightTally.', 'bright-tally'); ?></p>
                            <a href="<?php echo esc_url($docs_url); ?>" target="_blank" class="button button-primary">
                                <?php _e('Browse All Docs', 'bright-tally'); ?>
                            </a>
                        </div>
                    </div>
                    
                    <div style="margin-top: 40px; padding: 24px; background: #f0f6ff; border-radius: 8px; border-left: 4px solid #2271b1;">
                        <h3 style="margin-top: 0;"><?php _e('Need More Help?', 'bright-tally'); ?></h3>
                        <p><?php _e('Can\'t find what you\'re looking for? Our support team is here to help.', 'bright-tally'); ?></p>
                        <div style="margin-top: 15px;">
                            <a href="https://brighttally.com/support" target="_blank" class="button button-primary" style="margin-right: 10px;">
                                <?php _e('Contact Support', 'bright-tally'); ?>
                            </a>
                            <a href="mailto:hello@brighttally.com" class="button button-secondary">
                                <?php _e('Email Support', 'bright-tally'); ?>
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Register plugin settings
     *
     * @since 1.0.0
     */
    public function register_settings() {
        register_setting('bright_tally_settings', 'bright_tally_api_key', array(
            'sanitize_callback' => array($this, 'sanitize_api_key'),
            'default' => '',
        ));
        register_setting('bright_tally_settings', 'bright_tally_api_url', array(
            'sanitize_callback' => array($this, 'sanitize_api_url'),
            'default' => 'https://brighttally.com',
        ));
        register_setting('bright_tally_settings', 'bright_tally_default_theme', array(
            'sanitize_callback' => array($this, 'sanitize_theme'),
            'default' => 'light',
        ));
        register_setting('bright_tally_settings', 'bright_tally_cache_duration', array(
            'sanitize_callback' => array($this, 'sanitize_cache_duration'),
            'default' => 300,
        ));
        register_setting('bright_tally_settings', 'bright_tally_docs_url', array(
            'sanitize_callback' => array($this, 'sanitize_docs_url'),
            'default' => '',
        ));
    }
    
    /**
     * Sanitize API key
     *
     * @since 1.0.0
     * @param string $value The API key value
     * @return string Sanitized API key
     */
    public function sanitize_api_key($value) {
        $value = sanitize_text_field($value);
        // Validate API key format (alphanumeric, pipes, underscores, hyphens)
        if (!empty($value) && !preg_match('/^[a-zA-Z0-9|_-]+$/', $value)) {
            add_settings_error(
                'bright_tally_api_key',
                'invalid_api_key',
                esc_html__('Invalid API key format. Please check your API key.', 'bright-tally')
            );
            return '';
        }
        return $value;
    }
    
    /**
     * Sanitize API URL
     *
     * @since 1.0.0
     * @param string $value The API URL value
     * @return string Sanitized API URL
     */
    public function sanitize_api_url($value) {
        $value = esc_url_raw($value);
        // Validate URL format
        if (!empty($value) && !filter_var($value, FILTER_VALIDATE_URL)) {
            add_settings_error(
                'bright_tally_api_url',
                'invalid_api_url',
                esc_html__('Invalid API URL format. Please enter a valid URL.', 'bright-tally')
            );
            return 'https://brighttally.com';
        }
        // Ensure URL has protocol
        if (!empty($value) && !preg_match('/^https?:\/\//i', $value)) {
            $value = 'https://' . $value;
        }
        return $value;
    }
    
    /**
     * Sanitize theme
     *
     * @since 1.0.0
     * @param string $value The theme value
     * @return string Sanitized theme
     */
    public function sanitize_theme($value) {
        $allowed = array('light', 'dark');
        $value = sanitize_text_field($value);
        if (!in_array($value, $allowed, true)) {
            return 'light';
        }
        return $value;
    }
    
    /**
     * Sanitize cache duration
     *
     * @since 1.0.0
     * @param mixed $value The cache duration value
     * @return int Sanitized cache duration
     */
    public function sanitize_cache_duration($value) {
        $value = absint($value);
        // Ensure value is within reasonable bounds (60 seconds to 24 hours)
        if ($value < 60) {
            $value = 60;
        } elseif ($value > 86400) {
            $value = 86400;
        }
        return $value;
    }
    
    /**
     * Sanitize docs URL
     *
     * @since 1.0.0
     * @param string $value The docs URL value
     * @return string Sanitized docs URL
     */
    public function sanitize_docs_url($value) {
        $value = esc_url_raw($value);
        // Validate URL format if provided
        if (!empty($value) && !filter_var($value, FILTER_VALIDATE_URL)) {
            add_settings_error(
                'bright_tally_docs_url',
                'invalid_docs_url',
                esc_html__('Invalid documentation URL format. Please enter a valid URL.', 'bright-tally')
            );
            return '';
        }
        return $value;
    }
}
